/*******************************************************************************

    uBlock Origin Lite - a comprehensive, MV3-compliant content blocker
    Copyright (C) 2014-present Raymond Hill

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see {http://www.gnu.org/licenses/}.

    Home: https://github.com/gorhill/uBlock

*/

// ruleset: annoyances-cookies

// Important!
// Isolate from global scope

// Start of local scope
(function uBOL_trustedSetCookie() {

/******************************************************************************/

function trustedSetCookie(
    name = '',
    value = '',
    offsetExpiresSec = '',
    path = ''
) {
    if ( name === '' ) { return; }

    const safe = safeSelf();
    const logPrefix = safe.makeLogPrefix('set-cookie', name, value, path);
    const time = new Date();

    if ( value.includes('$now$') ) {
        value = value.replaceAll('$now$', time.getTime());
    }
    if ( value.includes('$currentDate$') ) {
        value = value.replaceAll('$currentDate$', time.toUTCString());
    }
    if ( value.includes('$currentISODate$') ) {
        value = value.replaceAll('$currentISODate$', time.toISOString());
    }

    let expires = '';
    if ( offsetExpiresSec !== '' ) {
        if ( offsetExpiresSec === '1day' ) {
            time.setDate(time.getDate() + 1);
        } else if ( offsetExpiresSec === '1year' ) {
            time.setFullYear(time.getFullYear() + 1);
        } else {
            if ( /^\d+$/.test(offsetExpiresSec) === false ) { return; }
            time.setSeconds(time.getSeconds() + parseInt(offsetExpiresSec, 10));
        }
        expires = time.toUTCString();
    }

    const done = setCookieFn(
        true,
        name,
        value,
        expires,
        path,
        safeSelf().getExtraArgs(Array.from(arguments), 4)
    );

    if ( done ) {
        safe.uboLog(logPrefix, 'Done');
    }
}

function safeSelf() {
    if ( scriptletGlobals.safeSelf ) {
        return scriptletGlobals.safeSelf;
    }
    const self = globalThis;
    const safe = {
        'Array_from': Array.from,
        'Error': self.Error,
        'Function_toStringFn': self.Function.prototype.toString,
        'Function_toString': thisArg => safe.Function_toStringFn.call(thisArg),
        'Math_floor': Math.floor,
        'Math_max': Math.max,
        'Math_min': Math.min,
        'Math_random': Math.random,
        'Object': Object,
        'Object_defineProperty': Object.defineProperty.bind(Object),
        'Object_defineProperties': Object.defineProperties.bind(Object),
        'Object_fromEntries': Object.fromEntries.bind(Object),
        'Object_getOwnPropertyDescriptor': Object.getOwnPropertyDescriptor.bind(Object),
        'Object_hasOwn': Object.hasOwn.bind(Object),
        'RegExp': self.RegExp,
        'RegExp_test': self.RegExp.prototype.test,
        'RegExp_exec': self.RegExp.prototype.exec,
        'Request_clone': self.Request.prototype.clone,
        'String': self.String,
        'String_fromCharCode': String.fromCharCode,
        'String_split': String.prototype.split,
        'XMLHttpRequest': self.XMLHttpRequest,
        'addEventListener': self.EventTarget.prototype.addEventListener,
        'removeEventListener': self.EventTarget.prototype.removeEventListener,
        'fetch': self.fetch,
        'JSON': self.JSON,
        'JSON_parseFn': self.JSON.parse,
        'JSON_stringifyFn': self.JSON.stringify,
        'JSON_parse': (...args) => safe.JSON_parseFn.call(safe.JSON, ...args),
        'JSON_stringify': (...args) => safe.JSON_stringifyFn.call(safe.JSON, ...args),
        'log': console.log.bind(console),
        // Properties
        logLevel: 0,
        // Methods
        makeLogPrefix(...args) {
            return this.sendToLogger && `[${args.join(' \u205D ')}]` || '';
        },
        uboLog(...args) {
            if ( this.sendToLogger === undefined ) { return; }
            if ( args === undefined || args[0] === '' ) { return; }
            return this.sendToLogger('info', ...args);
            
        },
        uboErr(...args) {
            if ( this.sendToLogger === undefined ) { return; }
            if ( args === undefined || args[0] === '' ) { return; }
            return this.sendToLogger('error', ...args);
        },
        escapeRegexChars(s) {
            return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        },
        initPattern(pattern, options = {}) {
            if ( pattern === '' ) {
                return { matchAll: true, expect: true };
            }
            const expect = (options.canNegate !== true || pattern.startsWith('!') === false);
            if ( expect === false ) {
                pattern = pattern.slice(1);
            }
            const match = /^\/(.+)\/([gimsu]*)$/.exec(pattern);
            if ( match !== null ) {
                return {
                    re: new this.RegExp(
                        match[1],
                        match[2] || options.flags
                    ),
                    expect,
                };
            }
            if ( options.flags !== undefined ) {
                return {
                    re: new this.RegExp(this.escapeRegexChars(pattern),
                        options.flags
                    ),
                    expect,
                };
            }
            return { pattern, expect };
        },
        testPattern(details, haystack) {
            if ( details.matchAll ) { return true; }
            if ( details.re ) {
                return this.RegExp_test.call(details.re, haystack) === details.expect;
            }
            return haystack.includes(details.pattern) === details.expect;
        },
        patternToRegex(pattern, flags = undefined, verbatim = false) {
            if ( pattern === '' ) { return /^/; }
            const match = /^\/(.+)\/([gimsu]*)$/.exec(pattern);
            if ( match === null ) {
                const reStr = this.escapeRegexChars(pattern);
                return new RegExp(verbatim ? `^${reStr}$` : reStr, flags);
            }
            try {
                return new RegExp(match[1], match[2] || undefined);
            }
            catch {
            }
            return /^/;
        },
        getExtraArgs(args, offset = 0) {
            const entries = args.slice(offset).reduce((out, v, i, a) => {
                if ( (i & 1) === 0 ) {
                    const rawValue = a[i+1];
                    const value = /^\d+$/.test(rawValue)
                        ? parseInt(rawValue, 10)
                        : rawValue;
                    out.push([ a[i], value ]);
                }
                return out;
            }, []);
            return this.Object_fromEntries(entries);
        },
        onIdle(fn, options) {
            if ( self.requestIdleCallback ) {
                return self.requestIdleCallback(fn, options);
            }
            return self.requestAnimationFrame(fn);
        },
        offIdle(id) {
            if ( self.requestIdleCallback ) {
                return self.cancelIdleCallback(id);
            }
            return self.cancelAnimationFrame(id);
        }
    };
    scriptletGlobals.safeSelf = safe;
    if ( scriptletGlobals.bcSecret === undefined ) { return safe; }
    // This is executed only when the logger is opened
    safe.logLevel = scriptletGlobals.logLevel || 1;
    let lastLogType = '';
    let lastLogText = '';
    let lastLogTime = 0;
    safe.toLogText = (type, ...args) => {
        if ( args.length === 0 ) { return; }
        const text = `[${document.location.hostname || document.location.href}]${args.join(' ')}`;
        if ( text === lastLogText && type === lastLogType ) {
            if ( (Date.now() - lastLogTime) < 5000 ) { return; }
        }
        lastLogType = type;
        lastLogText = text;
        lastLogTime = Date.now();
        return text;
    };
    try {
        const bc = new self.BroadcastChannel(scriptletGlobals.bcSecret);
        let bcBuffer = [];
        safe.sendToLogger = (type, ...args) => {
            const text = safe.toLogText(type, ...args);
            if ( text === undefined ) { return; }
            if ( bcBuffer === undefined ) {
                return bc.postMessage({ what: 'messageToLogger', type, text });
            }
            bcBuffer.push({ type, text });
        };
        bc.onmessage = ev => {
            const msg = ev.data;
            switch ( msg ) {
            case 'iamready!':
                if ( bcBuffer === undefined ) { break; }
                bcBuffer.forEach(({ type, text }) =>
                    bc.postMessage({ what: 'messageToLogger', type, text })
                );
                bcBuffer = undefined;
                break;
            case 'setScriptletLogLevelToOne':
                safe.logLevel = 1;
                break;
            case 'setScriptletLogLevelToTwo':
                safe.logLevel = 2;
                break;
            }
        };
        bc.postMessage('areyouready?');
    } catch {
        safe.sendToLogger = (type, ...args) => {
            const text = safe.toLogText(type, ...args);
            if ( text === undefined ) { return; }
            safe.log(`uBO ${text}`);
        };
    }
    return safe;
}

function setCookieFn(
    trusted = false,
    name = '',
    value = '',
    expires = '',
    path = '',
    options = {},
) {
    // https://datatracker.ietf.org/doc/html/rfc2616#section-2.2
    // https://github.com/uBlockOrigin/uBlock-issues/issues/2777
    if ( trusted === false && /[^!#$%&'*+\-.0-9A-Z[\]^_`a-z|~]/.test(name) ) {
        name = encodeURIComponent(name);
    }
    // https://datatracker.ietf.org/doc/html/rfc6265#section-4.1.1
    // The characters [",] are given a pass from the RFC requirements because
    // apparently browsers do not follow the RFC to the letter.
    if ( /[^ -:<-[\]-~]/.test(value) ) {
        value = encodeURIComponent(value);
    }

    const cookieBefore = getCookieFn(name);
    if ( cookieBefore !== undefined && options.dontOverwrite ) { return; }
    if ( cookieBefore === value && options.reload ) { return; }

    const cookieParts = [ name, '=', value ];
    if ( expires !== '' ) {
        cookieParts.push('; expires=', expires);
    }

    if ( path === '' ) { path = '/'; }
    else if ( path === 'none' ) { path = ''; }
    if ( path !== '' && path !== '/' ) { return; }
    if ( path === '/' ) {
        cookieParts.push('; path=/');
    }

    if ( trusted ) {
        if ( options.domain ) {
            let domain = options.domain;
            if ( /^\/.+\//.test(domain) ) {
                const baseURL = new URL(document.baseURI);
                const reDomain = new RegExp(domain.slice(1, -1));
                const match = reDomain.exec(baseURL.hostname);
                domain = match ? match[0] : undefined;
            }
            if ( domain ) {
                cookieParts.push(`; domain=${domain}`);
            }
        }
        cookieParts.push('; Secure');
    } else if ( /^__(Host|Secure)-/.test(name) ) {
        cookieParts.push('; Secure');
    }

    try {
        document.cookie = cookieParts.join('');
    } catch {
    }

    const done = getCookieFn(name) === value;
    if ( done && options.reload ) {
        window.location.reload();
    }

    return done;
}

function getCookieFn(
    name = ''
) {
    const safe = safeSelf();
    for ( const s of safe.String_split.call(document.cookie, /\s*;\s*/) ) {
        const pos = s.indexOf('=');
        if ( pos === -1 ) { continue; }
        if ( s.slice(0, pos) !== name ) { continue; }
        return s.slice(pos+1).trim();
    }
}

/******************************************************************************/

const scriptletGlobals = {}; // eslint-disable-line
const argsList = [["euconsent-v2","CPt3fQAPt3fQACNAFAENDLCgAAAAAAAAACiQAAAOCgDAB-AIsAZ8A6QDBAHBAAAA.YAAAAAAAAAAA","1year"],["tracking-opt-in-status","rejected","1year"],["addtl_consent","1~","1year"],["consentUUID","4828f4be-a6c6-46d4-b46a-5a812a8c0f81_35"],["dm-euconsent-v2","CQEXy8AQEXy8ABpAIAENAtEgAAAAAEPgAAqIAABBqALMNCogD7IkJCLQMIoEAKgrCAigQAAAAkDRAQAmDAp2BgEusJEAIAUAAwQAgABRkACAAASABCIAJACgQAAQCBQAAgAACAQAMDAAGAC0EAgABAdAhTAggUCwASMyIhTAhCgSCAlsoEEgCBBXCEIs8CCAREwUAAAJABWAAICwWAxJICViQQJcQbQAAEACAQQgVCKTowBDAmbLVXiibRlaQFoAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAgAAAIAA.YAAACfwAAAAA"],["fig_save_consent","iTTPgpSWqAGGcd3vV88zNDbHsABxE1hB","1year"],["euconsent-v2","CPubvkAPubvkAAHABBENDMCgAAAAAAAAAB5YAAAAAAAA.YAAAAAAAAAAA","1year"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0008%3A0","1year"],["PUR_SUBSCRIPTION","PREMIUM"],["CookieConsent","{necessary:true%2Cpreferences:false%2Cstatistics:false%2Cmarketing:false}","1year"],["cb","{\"version\":\"1\",\"date\":\"2029_07_03\",\"third_party_consent\":\"withdrawn\",\"categories\":{},\"integration_controls\":{}}","","","reload","1"],["datr","mWTaZBxAoW8lFl0v3EpECGYi","1year","","reload","1"],["myscript","{%22acceptance%22:true%2C%22analytics%22:false}"],["tm_cookie_setting","Analytics","","","reload","1"],["BCP","AD=0&AL=0&SM=0"],["cookies","{\"system\":1,\"target\":0}"],["CookieControl","{\"interactedWith\":true,\"acceptedCategories\":[\"essentials\"]}"],["trackingPermissionConsentsValue","%7B%22cookies_analytics%22%3Afalse%2C%22cookies_personalization%22%3Afalse%2C%22cookies_advertisement%22%3Afalse%7D"],["pfUserCookPolicy","12562660:1"],["viucom-cookie-accept","[\"technical_necessary\"]"],["BN.User.CookieConsent","CAE%3D"],["CookieConsent","{stamp:%27BvqlZ34xqOpzSOaxY1VgwOy5CZ1lfC4Xs2JNiDqz9M4rNuyyAY56Jg==%27%2Cnecessary:true%2Cpreferences:false%2Cstatistics:false%2Cmarketing:false%2Cmethod:%27explicit%27%2Cver:1%2Cutc:1697932000537}","1year"],["CookieConsent","{stamp:%27Uv9YOAVP5djCBw71lxhE4rNAfTObaGck2Sn3rKWf9dPCYpqcWvAlpA==%27%2Cnecessary:true%2Cpreferences:false%2Cstatistics:false%2Cmarketing:false%2Cmethod:%27explicit%27%2Cver:2}","1year"],["CookieConsent","{stamp:%27Uv9YOAVP5djCBw71lxhE4rNAfTObaGck2Sn3rKWf9dPCYpqcWvAlpA==%27%2Cnecessary:true%2Cpreferences:false%2Cstatistics:false%2Cmarketing:false%2Cmethod:%27explicit%27%2Cver:3}","1year"],["CookieConsent","{stamp:%27fbbiPQWa8SRQm47Ge8hRVOAPAgw2jXlG8o/+hp1euVVi1qtFQj1O3w==%27%2Cnecessary:true%2Cpreferences:true%2Cstatistics:true%2Cmarketing:true%2Cmethod:%27explicit%27%2Cver:4}","1year"],["CookieConsent","{stamp:%27Uv9YOAVP5djCBw71lxhE4rNAfTObaGck2Sn3rKWf9dPCYpqcWvAlpA==%27%2Cnecessary:true%2Cpreferences:false%2Cstatistics:false%2Cmarketing:false%2Cmethod:%27explicit%27%2Cver:5}","1year"],["CookieConsent","{stamp:%27Uv9YOAVP5djCBw71lxhE4rNAfTObaGck2Sn3rKWf9dPCYpqcWvAlpA==%27%2Cnecessary:true%2Cpreferences:false%2Cstatistics:false%2Cmarketing:false%2Cmethod:%27explicit%27%2Cver:6}","1year"],["SK.Website.ConsentManager","W1siYmFzZSIsdHJ1ZV0sWyJnb29nbGVhbmFseXRpY3MiLGZhbHNlXSxbIm1ldGFwaXhlbCIsZmFsc2VdXQ=="],["consent","%7B%22level%22%3A%5B%22necessary%22%5D%2C%22revision%22%3A0%2C%22data%22%3A%7B%22id%22%3A%22DarYF7gx7v%22%7D%2C%22rfc_cookie%22%3Atrue%7D"],["CookieConsents","{\"StrictlyNecessary\":true,\"Analytical\":false,\"Performance\":false,\"AdvertisingAndTargeting\":false,\"timestamp\":\"\"}"],["cookie-consent-news-3","{\"optedIn\":true,\"statistics\":false,\"functionality\":false}"],["cr_consent","2|tracking=false+analytics=false+marketing=false","","","reload","1"],["hbl_cookie_prefs_v1","[%22Essential%22]"],["mmc-cookie-consent","{\"Id\":\"c2457eddbecc20cbab3869374900e864\",\"CategoryID\":4}","1year"],["fixami.app_state","eyJzdGF0ZSI6eyJpc1RheEluY2x1ZGVkIjp0cnVlLCJpc0ZpcnN0VmlzaXQiOnRydWUsImlzQ3VzdG9tZXJTZXJ2aWNlTW9kYWxPcGVuIjpmYWxzZSwiYWNjZXB0ZWRDb29raWVzIjoiMCIsImhhc0FjY2VwdGVkQ29va2llcyI6dHJ1ZSwiaGFzQWNjZXB0ZWRNYXJrZXRpbmdDb29raWVzIjp0cnVlLCJoYXNBY2NlcHRlZFN0YXRpc3RpY3NDb29raWVzIjp0cnVlfSwidmVyc2lvbiI6MH0="],["website_cookies_bar","{\"required\": true, \"optional\": false}"],["FCCDCF","%5Bnull%2Cnull%2Cnull%2C%5B%22CP2G1QAP2G1QAEsACDENAdEgAAAAAAAAAAwIAAAGfgAgGfAA%22%2C%221~%22%2C%22EA7F77A0-0F26-47A6-A8A4-74D65175A664%22%5D%2Cnull%2Cnull%2C%5B%5D%5D"],["euconsent-v2","CP2KIMAP2KIMAAHABBENAcEgAAAAAAAAAAiQAAAAAAEEoAMAARBqDQAYAAiDUKgAwABEGopABgACINQ6ADAAEQaiEAGAAIg1BIAMAARBqGQAYAAiDUAA.YAAAAAAAAAAA","1year","","domain","softonic.com"],["euconsent-v2","CP2KIMAP2KIMAAHABBENAcEgAAAAAAAAAAiQAAAAAAEEoAMAARBqDQAYAAiDUKgAwABEGopABgACINQ6ADAAEQaiEAGAAIg1BIAMAARBqGQAYAAiDUAA.YAAAAAAAAAAA","1year","","domain","softonic.pl"],["swapcard-cookie-consent","%7B%22accepted%22%3Afalse%7D"],["cookies-consent","agree"],["cookie_consent","100","1year"],["consent_setting","analytics%3A0%7Cfunctional%3A1%7Cmarketing%3A0","","","reload","1"],["cookieConsent","functional=1&analytics=0&marketing=0","","","reload","1"],["euconsent-v2","CP20-YAP20-YAAKAyBENAfEgAAAAAAAAAAwIAAAI8gBAGfAR4AAA.YAAAAAAAAAAA","1year"],["consentUUID","cefc5c9f-0e4c-4d6f-ad00-41a1e49409ed_27_29"],["OptanonAlertBoxClosed","$currentDate$","1year"],["cookie_consent","necessary%3A1%2Cstatistics%3A0%2Cmarketing%3A0"],["FTConsent","marketingBypost%3Aoff%2CmarketingByemail%3Aoff%2CmarketingByphonecall%3Aoff%2CmarketingByfax%3Aoff%2CmarketingBysms%3Aoff%2CenhancementBypost%3Aoff%2CenhancementByemail%3Aoff%2CenhancementByphonecall%3Aoff%2CenhancementByfax%3Aoff%2CenhancementBysms%3Aoff%2CbehaviouraladsOnsite%3Aoff%2CdemographicadsOnsite%3Aoff%2CrecommendedcontentOnsite%3Aon%2CprogrammaticadsOnsite%3Aoff%2CcookiesUseraccept%3Aoff%2CcookiesOnsite%3Aoff%2CmembergetmemberByemail%3Aoff%2CpermutiveadsOnsite%3Aoff%2CpersonalisedmarketingOnsite%3Aoff","","","domain","ft.com"],["cookie_preferences","{%22marketing%22:false%2C%22analytics%22:false}"],["gsbbanner","closed"],["consent","%7B%22functional%22%3A%7B%22consent%22%3Atrue%7D%2C%22analytical%22%3A%7B%22consent%22%3Afalse%7D%2C%22personalization%22%3A%7B%22consent%22%3Afalse%7D%2C%22geolocation%22%3A%7B%22consent%22%3Afalse%7D%2C%22ads%22%3A%7B%22consent%22%3Afalse%7D%7D"],["server_consent","%7B%22functional%22%3A%7B%22consent%22%3Atrue%7D%2C%22analytical%22%3A%7B%22consent%22%3Afalse%7D%2C%22personalization%22%3A%7B%22consent%22%3Afalse%7D%2C%22geolocation%22%3A%7B%22consent%22%3Afalse%7D%2C%22ads%22%3A%7B%22consent%22%3Afalse%7D%7D"],["server_addtl_consent","1~"],["LEGO_COOKIE_SETTINGS","{\"preferences\":[{\"key\":\"necessary\",\"value\":true},{\"key\":\"analytics\",\"value\":false},{\"key\":\"legomarketing\",\"value\":false},{\"key\":\"thirdparty\",\"value\":false}]}"],["LEGO_COOKIE_SETTINGS","{\"preferences\":[{\"key\":\"analytics\",\"value\":false},{\"key\":\"legomarketing\",\"value\":false},{\"key\":\"necessary\",\"value\":true},{\"key\":\"thirdparty\",\"value\":false}]}"],["LEGO_OPT_OUT_SETTINGS","{\"preferences\":[{\"key\":\"doNotAdvertiseBasedOnMyPersonalData\",\"value\":true}]}"],["cookie_preferences","{%22performance%22:false%2C%22marketing%22:false%2C%22necessary%22:true}"],["veriff_cookie_consent","%7B%22performance%22%3Afalse%2C%22functionality%22%3Afalse%2C%22targeting%22%3Afalse%7D"],["veriff_cookie_notice","%7B%22dismisssed%22%3Atrue%7D"],["USE_COOKIE_CONSENT_STATE","{%22necessary%22:true}"],["consentCookie","{\"arena-functional\":true,\"sentry\":false,\"cloudflare\":true,\"google-tag-manager\":false,\"google-analytics\":false,\"facebook-ads\":false,\"google-ads\":false,\"bing-ads\":false,\"arena-advertising\":false}","","","domain","arena.pl"],["cookieConsent","technical=true&analytics=false&advertising=false"],["acceptTracking","{\"technical\":true,\"analytics\":false,\"advertising\":false}"],["gdprConsent","%7B%22adv%22%3Afalse%2C%22func%22%3Afalse%2C%22tech%22%3Afalse%7D","","","domain","deal.by"],["cookie_preference","functional","","","domain","zwenkwielen.net"],["__consent","%5B%22required%22%5D"],["sncc","P%3D15%3AV%3D33.0.0%26C%3DC01%26D%3Dtrue","","","domain","biomedcentral.com"],["sncc","P%3D17%3AV%3D54.0.0%26C%3DC01%26D%3Dtrue","","","domain","link.springer.com"],["sncc","P%3D8%3AV%3D77.0.0%26C%3DC01%26D%3Dtrue","","","domain","nature.com"],["sncc","P%3D49%3AV%3D60.0.0%26C%3DC01%26D%3Dtrue","","","domain","springernature.com"],["sncc","P%3D65%3AV%3D14.0.0%26C%3DC01%26D%3Dtrue","","","domain","palgrave.com"],["sncc","P%3D71%3AV%3D9.0.0%26C%3DC01%26D%3Dtrue","","","domain","apress.com"],["bolConsentChoices","source#OFC|version#6|int-tran#false|ext-tran#false|int-beh#false|ext-beh#false","","","domain","www.bol.com","reload","1"],["cookiePolicy","{\"accepted\":true,\"technical\":true,\"statistics\":false,\"marketing\":false,\"expire\":$now$}","","","domain","abw.by"],["cookie_consent","{\"purposes\":{\"marketing\":false,\"analytics\":false}}"],["cookiesjsr","%7B%22functional%22%3Atrue%2C%22analytics%22%3Afalse%2C%22facebook_pixel%22%3Afalse%2C%22iframe%22%3Atrue%2C%22video%22%3Atrue%2C%22facebook_messenger%22%3Afalse%7D"],["cpmt_xa","16200","","","domain","open.online"],["euconsent-v2","CQBYokAQBYokADaAAAENA7EgAAAAAAAAABCYAAAAAADBIMIACAAFwAUABUADgAHgAQAAvgBkAGoAPAAiABMACqAG8APQAfgBCQCGAIkARwAlgBNADAAGHAMoAywBsgDngHcAd8A9gD4gH2AfsA_wEAAIpARcBGACNQEiASWAn4CgwFQAVcAuYBegDFAGiANoAbgA4gCHYEegSKAnYBQ4CjwFIgLYAXIAu8BeYDBgGGwMjAyQBk4DLgGZgM5gauBrIDYwG0ANvAbmA3UBwQDkwHLgPHAe0A_4CCYEGAIQwQtBC8CHMEPQQ_gj6CP4EigJIQSYBJkCWYEtwJfATAgmWCZgEzgJqgTcAnMBOkCdwE8AKEAULAoqBSUlA7AAQAAsACgAHAAeABEACYAFUALgAYoBDAESAI4AUYAwABsgDvAH5AVEBVwC5gGKAOoAhABDoCJgEXwI9AkUBR4CmgFigLYAXnAyMDJAGTgM5AawA28B7QEAQIHgQYAhCBD0CPoEigJIASVAl0BL6CZYJmATOAmqBNwCcIE5gJ3ATwAnmBRUCko6DsAAuACgAKgAcABBAC4AL4AZABqADwAIgATAApgBVgC4ALoAYgA3gB6AD9AIYAiQBLACaAE4AKMAYAAwwBlADRAGyAOeAdwB3wD2APiAfYB-gD_gIoAjEBHQElgJ-AoMBUQFXALEAXOAvIC9AGKANoAbgA4gB1AD7AIQAQ6Ai-BHoEigJkATsAoeBR4FIAKaAVYAsWBbAFsgLdAXAAuQBdoC7wF5gL6AYMAw0Bj0DIwMkAZOAyqBlgGXAMzAZyA00BqsDVwNYAbQA28BuoDiwHJgOXAeOA8kB7QD6wH3AP7Af8BAECDAELYIcgh0BD2CPoI_gSKAkgBJkCWYEugJfATAAmYBM4CaoE3AJvAThAnMBOmCdoJ3ATwAnmBQACgwFCwKJAUVApKQgaAALAAoAC4AGoATAAqgBcADEAG8APQAjgBgADngHcAd4A_wCKAEpAKDAVEBVwC5gGKANoAdQBHoCmgFWALFAWiAuABcgDIwGTgM5AaqA8cB_YEGAIUAQtAh0BD0CRQEkAJdATOAnMBO4CeAE8wKDAUVApKUgtAALgAoACoAHAAQQAyADUAHgARAAmABSACqAGIAP0AhgCJAFGAMAAZQA0QBsgDnAHfAPsA_QCLAEYgI6AkoBQYCogKuAXMAvIBigDaAG4AOoAe0A-wCHQETAIvgR6BIoCdgFDgKQAU0AqwBYoC2AFwALkAXaAvMBfQDDYGRgZIAyeBlgGXAM5gawBrIDbwG6gOCAcmA8UB44D2gH9gP-AgmBBgCEIELQIZwQ5BDqCPoI_gSKAkhBJgEmQJZgS6Al8BMACZgEzgJqgTeAnMBO4CeAE8wKFgUVApKWgIgA1AEcAMAAdwBegD7AKaAVYAzMB44EPQJmATcAnMBO4CeAFJA.YAAAAAAAAAAA","","","domain","open.online"],["euconsent","CQB2TAAQB2TAAGRABAENA8EgAAAAAAAAAAAAAAAUrgAAAAAA.YAAAAAAAAAAA,","","","domain","ilovefreegle.org"],["cookieyes-consent","consent:no,action:yes,necessary:yes,functional:no,analytics:no,performance:no,advertisement:no,other:no","","","domain","ilovefreegle.org"],["CONSENTMGR","c14:1|c1:0|c2:0|c3:0|c4:0|c5:0|c6:0|c7:0|c8:0|c9:0|c10:0|c11:0|c12:0|c13:0|c15:0|ts:1721686992938|consent:true","","","domain","eurostar.com"],["ng-cc-accepted","accepted","","","domain","iskon.hr"],["ng-cc-analytics","0","","","domain","iskon.hr"],["ng-cc-marketing","0","","","domain","iskon.hr"],["user-cookie-preferences","{%22necessary-cookies%22:true%2C%22marketing-advertising-cookies%22:false%2C%22functional-cookies%22:false%2C%22analytics-cookies%22:false}","","","domain","virgin.com"],["TVPtcs22","CQDpp0AQDpp0AFfADAENBCFgAAAAAAAAAB5YAAAUvAJMNCogDLIkJCDQMIIEAKgrCAigQAAAAkDRAQAmDAp2BgEusJEAIAUAAwQAgABRkACAAASABCIAIACgQAAQCBQABgAQDAQAMDAAGACwEAgABAdAhTAggUCwASMyIhTAhCASCAlsqEEgCBBXCEIs8CCAREwUAAAJABWAAICwWBxJICViQQJcQbQAAEACAQQAVCKTswBBAGbLUXiybRlaQFo-YLntMAwAAAAA.YAAAAAAABYAAAAAA"],["CONSENTMGR","c1:1%7Cc2:0%7Cc3:0%7Cc4:0%7Cts:$now$%7Cconsent:true"],["cookiePolicyAcceptedDate","$currentISODate$"],["OptanonConsent","groups=C0004%3A0%2CC0005%3A0%2CC0003%3A1%2CC0002%3A0%2CC0001%3A1%2CV2STACK42%3A1","","","domain","blick.ch"],["OptanonAlertBoxClosed","$currentDate$"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A0%2CC0004%3A0%2CC0005%3A0","","","domain","olympics.com"],["visme_cookies_settings","{\"analytics\":false,\"advertisement\":false,\"functionality\":false}","","","domain","visme.co"],["visme_cookies_use","no","","","domain","visme.co"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0004%3A0%2CC0003%3A0","","","domain","ted.com"],["userConsent","%7B%22marketing%22%3Afalse%2C%22version%22%3A%225%22%7D","","","domain","lebenslauf.com"],["userConsent","%7B%22marketing%22%3Afalse%2C%22version%22%3A%225%22%7D","","","domain","xing.com"],["ac","{\"essential\":true,\"functional\":false}"],["cookie_4_cookie","%7B%22onswingers%22%3Atrue%2C%22google-analytics%22%3Afalse%7D","","","domain","onswingers.com"],["obiConsent","c1-1|c2-0|c3-0|c4-0|c5-0|ts-$now$|consent-true"],["OptanonConsent","isMarketing=0&isStatistic=0"],["cookiesSettingsData","{\"1\":2,\"2\":0,\"3\":0,\"4\":0}"],["cc","{%22disagreement%22:[%22ad%22%2C%22visit%22]%2C%22creation%22:$now$%2C%22update%22:$now$}","","","domain","mediapart.fr"],["COOKIE_CONSENT","%7B%22ANALYTICS%22%3Afalse%7D","","","domain","zus.pl"],["COOKIE_CONSENT_EZUS","{\"FUNCTIONAL\":true,\"ANALYTICS\":false}","","","domain","zus.pl"],["CONSENTMGR","consent:true%7Cts:$now$%7Cc1:0%7Cc3:0%7Cc5:1%7Cc6:0%7Cc7:0%7Cc8:0%7Cc9:1","","","domain","ups.com"],["HASSEENRECNOTICE","TRUE","","","domain","ups.com"],["OptanonConsent","groups=C0003%3A0%2CC0004%3A0%2CC0002%3A0%2CC0001%3A1","","","domain","www.saal-digital.de"],["OptanonAlertBoxClosed","$currentDate$","","","domain","www.saal-digital.de"],["cookieOptions","{\"analitics\":false,\"marketing\":false}"],["cookieconsent_status","accept_selected"],["cookieconsent_preferences_disabled","functionality,analytics,marketing"],["brandcrowd-cookie-consent","%7B%22advertisingStorage%22%3Afalse%2C%22analyticsStorage%22%3Afalse%2C%22advertisingUserData%22%3Afalse%2C%22advertisingPersonalization%22%3Afalse%7D"],["ka3kaaa","$now$#REFUSE_ALL"],["rev_cookie_consent","{%22preferences%22:false%2C%22analytics%22:false%2C%22advertising%22:false}","","","domain","revolut.com"],["FINNAIR_COOKIE_CONSENTS","W3siY29uc2VudFRleHRJZCI6ImYtY29va2llcy1hbmFseXRpY3MiLCJlZmZlY3RpdmVGcm9tIjoxNzAxNjQwODAwMDAwLCJjb25zZW50U3RhdHVzIjoiZGVjbGluZWQifSx7ImNvbnNlbnRUZXh0SWQiOiJmLWNvb2tpZXMtcGVyc29uYWxpemF0aW9uIiwiZWZmZWN0aXZlRnJvbSI6MTUyMjUzMDAwMDAwMCwiY29uc2VudFN0YXR1cyI6ImRlY2xpbmVkIn0seyJjb25zZW50VGV4dElkIjoiZi1jb29raWVzLWFkdmVydGlzaW5nIiwiZWZmZWN0aXZlRnJvbSI6MTcwODYzOTIwMDAwMCwiY29uc2VudFN0YXR1cyI6ImRlY2xpbmVkIn1d"],["FINNAIR_COOKIE_ID","l-d3a86210-7bdf-41eb-bc29-9c233e0f4c9c"],["cookie_level","%7B%22necessary%22%3A%221%22%2C%22performance%22%3A%220%22%2C%22remarketing%22%3A%220%22%7D"],["gdpr","-ga-ads-fbp-htj-yt","2629743"],["a9_cookie_control_prefs_v2","%255B%2522essential%2522%255D"],["SK_CONSENT","{%22hasAccepted%22:true%2C%22functional%22:false%2C%22necessary%22:true%2C%22statistics%22:false}","","","domain","skandia.se"],["ConsentChecked","{\"userHasSetCookies\":true,\"functionalityCookies\":false,\"statisticCookies\":false,\"marketingCookies\":false}"],["cc_cookie","%7B%22level%22%3A%5B%22necessary%22%5D%2C%22revision%22%3A0%2C%22data%22%3Anull%2C%22rfc_cookie%22%3Atrue%7D","1year"],["gcbca","","1day"],["gcbpl","p1|st1|pt1","1year"],["GHCC","Required:1-Analytics:0-SocialMedia:0-Advertising:0","","","domain","github.com"],["cookify","eyJuZWNlc3NhcnkiOnRydWUsIm1hcmtldGluZyI6ZmFsc2UsInZpZXdlZCI6dHJ1ZX0=","1year"],["eduskunta_cookie_consent","cookie-consent_rejected","5259487"],["cookiehub","eyJhbnN3ZXJlZCI6dHJ1ZSwicmV2aXNpb24iOjEsImRudCI6ZmFsc2UsImFsbG93U2FsZSI6dHJ1ZSwiaW1wbGljdCI6ZmFsc2UsInJlZ2lvbiI6IiIsInRva2VuIjoiUzZqU1p4bmtZcW5QaENqN3QxbkR4SXlkTnZtbFYxY2tmN1R6dXBhekQ1RWZoZjZ5MU45WHYxdnd6c2Z5UXJEMSIsInRpbWVzdGFtcCI6IjIwMjUtMDMtMjJUMjM6MDE6MjYuNDk0WiIsImFsbEFsbG93ZWQiOmZhbHNlLCJjYXRlZ29yaWVzIjpbXSwidmVuZG9ycyI6W10sInNlcnZpY2VzIjpbXSwiaW1wbGljaXQiOmZhbHNlfQ==","1year"],["gdpr_consent","necessary%3D1%26analytics%3D0","","","domain","tvprofil.com","reload","1"],["cookiesjsr","%7B%22functional%22%3Afalse%2C%22analytics%22%3Afalse%2C%22facebook_pixel%22%3Afalse%2C%22recaptcha%22%3Afalse%2C%22gtag%22%3Afalse%2C%22video%22%3Afalse%7D"],["cookieconsent","saved=1&functional=0&analytics=0&marketing=0","","","reload","1"],["cookie_preferences_set_v1","%7B%22state%22%3A%7B%22preferences%22%3A%7B%22necessary%22%3Atrue%2C%22functional%22%3Afalse%2C%22performance%22%3Afalse%2C%22targeting%22%3Afalse%2C%22userHasMadeChoice%22%3Atrue%7D%2C%22functionalEnabled%22%3Afalse%2C%22performanceEnabled%22%3Afalse%2C%22targetingEnabled%22%3Afalse%7D%2C%22version%22%3A0%7D","","","domain",".kick.com"],["euconsent-v2","CQULowAQULowAAHABBENBxFgAAAAAAAAAAiQAAAAAACkoAMAAQWrKQAYAAgtWOgAwABBashABgACC1YSADAAEFqw.YAAAAAAAAAAA","1year","","domain","filehippo.com"],["euconsent-v2","CQULowAQULowAAHABBENBxFgAAAAAAAAAAiQAAAAAACkoAMAAQWrKQAYAAgtWOgAwABBashABgACC1YSADAAEFqw.YAAAAAAAAAAA","1year","","domain","filehippo.de"],["euconsent-v2","CQULowAQULowAAHABBENBxFgAAAAAAAAAAiQAAAAAACkoAMAAQWrKQAYAAgtWOgAwABBashABgACC1YSADAAEFqw.YAAAAAAAAAAA","1year","","domain","filehippo.jp"],["euconsent-v2","CQULowAQULowAAHABBENBxFgAAAAAAAAAAiQAAAAAACkoAMAAQWrKQAYAAgtWOgAwABBashABgACC1YSADAAEFqw.YAAAAAAAAAAA","1year","","domain","filehippo.pl"],["elli_cookie_consent","[\"ESSENTIAL\"]","1year"],["cookiehub","eyJhbnN3ZXJlZCI6dHJ1ZSwicmV2aXNpb24iOjEsImRudCI6ZmFsc2UsImFsbG93U2FsZSI6ZmFsc2UsImltcGxpY3QiOmZhbHNlLCJyZWdpb24iOiIiLCJ0b2tlbiI6IlI1T3BFRU1NQTFmWGJQcHZVZ2Rnbm5PNGROMjJMejBQUzN2ajA0T0dHZFpTckd2RW9GVlpRVkpUSklUNElRbVYiLCJ0aW1lc3RhbXAiOiIyMDI1LTA3LTA5VDExOjMzOjEyLjAwNVoiLCJhbGxBbGxvd2VkIjpmYWxzZSwiY2F0ZWdvcmllcyI6W10sInZlbmRvcnMiOltdLCJzZXJ2aWNlcyI6W10sImltcGxpY2l0IjpmYWxzZX0=","1year"],["nwc","!adsense=false!gajs=false!addthis=false","1year"],["eupubconsent-v2","CQUX_xAQUX_xAAcABBENByFgAAAAAAAAABpYAAAAAAGhAAgADgBOAP3AhWEgRAALAAqABwADwAIAAZAA0AB4AEwAKQAVQA3gB-AEIAI4ATQAwwBlADnAHcAP0Af4BJQC-gHtASIAnYBQ4C2AF5gMuAbGBBgCF4qAGAEwARwB7RkAMAJgAjgD2jgA0ADgAPADQAI4AhAD9wIVjoEYACwAKgAcABAADIAGgAPAAmABSACqAG8APwAmgBOADDAGUANEAc4A7gB-gD_AIsAR0AkoB7QEWAIvASIAnYBQ4C2AF5gMsAZcBBggADAAeAGgCFZCAaAAsAJgAVQA3gCOAHOAO4Af4BfQD2gIWARYBBgkADAHcAywB-5KAcAAsADgAPAAmABSACqAI4BCwCLAEXgJEAWwAvMCDBQASAI4ATgA7gD_gKkAVkA_cCFYELykB4ABYAFQAOAAgABkADQAHgATAApABVAD8AMoAaIA5wB-gEWAI6ASUAuoB7QELAIvASIAnYBQ4C2AF5gMsAZcBBgsAYAAeABkAFQAQgAkABsAEcAKQAdwBRQCmgF2AMIAZwA04B1QD2gIFAQsAtgBj4DYwICgQwA.YAAAAAAAAAAA","1year"],["_gdprConsents","essentials:true|analytics:false|preferences:false|marketing:false","1year"],["p_cc","min","1year"],["apex__cookiePreferences","Operational"],["website_cookies_bar","{\"required\": true, \"optional\": false}","1year"],["amcookie_allowed","1","1year","","domain","www.vpd.fi"],["amcookie_disallowed","recently_viewed_product%2Crecently_viewed_product_previous%2Crecently_compared_product%2Crecently_compared_product_previous%2C_ga%2C_gid%2C_gat","1year","","domain","www.vpd.fi"],["cookie-consent","{\"categories\":[\"analytics\"],\"level\":[\"analytics\"],\"revision\":0,\"data\":null,\"rfc_cookie\":false,\"consent_date\":\"$currentISODate$\",\"consent_uuid\":\"738928d9-c950-46d2-a732-700c986cf81a\",\"last_consent_update\":\"$currentISODate$\"}","1year"],["euconsent-v2","CQXekwAQXekwAAHABBENB7FgAAAAAAAAACQgAAAAAAAA.YAAAAAAAAAAA"],["cookie_consent","eyJpdiI6ImwwWXVzYitiUmpWSDJmMWtkdHMvanc9PSIsInZhbHVlIjoicERPYnN6akRVbCttQVBaSHhMTzd3TW4xZGJTVWxqK2JwTlNYWjhlZ29JanVXOWdQU2x5RVhaRVpuR0l3QjBYOSIsIm1hYyI6ImM4YmU4OTYzZjYwODEyMmRlMmFmZjU1ZmE0YjcyZDA0ZmQ2MmEyOWZhNjA5NzA4ZGQwOWY2NzgyNjEwOTBhMDAiLCJ0YWciOiIifQ%3D%3D","1year","","reload","1"],["cookiebar","CookieDisallowed"],["_EVAcookieaccept","Y","1year"],["_EVAGDPRfunctional","Y","1year"],["OptanonConsent","groups=C0004%3A0%2CC0003%3A1%2CC0002%3A0%2CC0001%3A1%2CSTACK42%3A0","1year"],["eupubconsent-v2","CPt6LrpPt6LrpAcABBENDKCgAAAAAAAAAAYgGBtX_T5eb2vj-3ZcN_tkaYwP55y3o2wzhhaIke8NwIeH7BoGJ2MwvBV4JiACGBAkkiKBAQVlHGBcCQAAgIgRiSKMYk2MjzNKJLJAilMbO0NYCD9mnkHT2ZCY70-uO__zvneAAAAYJABAXmKgAgLzGQAQF5joAIC8yUAEBeZSACAvMAAA.YAAAAAAAAAAA","1year","","reload","1"],["OptanonConsent","groups=1%3A1%2C2%3A0%2C3%3A1%2C4%3A0%2C5%3A1%2CBG57%3A0%2CBG58%3A0%2CBG59%3A0","1year"],["consentUUID","84561898-f8ab-4872-8ae3-b46523df21ba_38","1year"],["euconsent-v2","CQISSMAQISSMAAGABBENBOFsAP_gAAAAAApAAAAAAAAA.YAAAAAAAAAAA","1year"],["consentDate","$currentISODate$","1year"],["gravitoData","{\"TCString\":\"CQJXf4AQJXf4ABUASAFIBTFgAAAAAAAAAApAAAAAAAChIBoACwAKgAeABBADIANAAiABMAD8AIQAbIBIgDFALzHQDwAFgAVABBADIANAAiABMAD8ANEAbIBFgCRAGKAOoAmQBeZCAGAAsAxQB1CUAcABYARAAmADZAMUAdQBeZSAcAAsACoAIIAZABoAEQAJgAfgBogDZAIsASIAxQB1AF5g.YAAAAAAAAIAA\",\"NonTCFVendors\":[{\"id\":1,\"name\":\"Facebook\",\"consent\":true},{\"id\":3,\"name\":\"Google\",\"consent\":true},{\"id\":9,\"name\":\"Twitter\",\"consent\":true},{\"id\":10,\"name\":\"Wordpress\",\"consent\":true},{\"id\":15,\"name\":\"Linkedin\",\"consent\":true},{\"id\":19,\"name\":\"Vimeo\",\"consent\":true},{\"id\":27,\"name\":\"Apple\",\"consent\":true}]}","1year"],["TcString","CQJXf4AQJXf4ABUASAFIBTFsAP_gAAAAAApAAAAAAAChIBoACwAKgAeABBADIANAAiABMAD8AIQAbIBIgDFALzHQDwAFgAVABBADIANAAiABMAD8ANEAbIBFgCRAGKAOoAmQBeZCAGAAsAxQB1CUAcABYARAAmADZAMUAdQBeZSAcAAsACoAIIAZABoAEQAJgAfgBogDZAIsASIAxQB1AF5g.YAAAAAAAAIAA","1year"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A1%2CC0003%3A1%2CC0004%3A0%2CC0005%3A0","1year"],["ladies-cookies-overlay","%7B%22cookie-category-essential%22%3Atrue%2C%22cookie-category-stats%22%3Afalse%2C%22cookie-category-map_services%22%3Atrue%7D","","","reload","1"],["opt_out","analyse,werbe"],["OptanonConsent","groups=C0001%3A1%2CC0003%3A1%2CSPD_BG%3A1%2CC0002%3A1%2CC0004%3A1%2CC0005%3A1","","","reload","1"],["STYXKEY_your_privacy_settings","%7B%22strict%22%3A%221%22%2C%22thirdparty%22%3A%221%22%2C%22advanced%22%3A%220%22%7D","1year","","reload","1"],["OptanonConsent","groups=C0001%3A1%2CC0009%3A0%2CC0002%3A0%2CC0003%3A1%2CC0004%3A1","1year"],["allowCookies","{\"uvc\":true,\"__cfduid\":true}"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A1%2CC0004%3A0%2CC0005%3A1","1year"],["consents",":4:6:7:8:9:10:11:12:13:19:"],["__cmpcpc","__1_2__"],["__cmpcvc","__c24599_s94_c24102_s40_s1052_s65_c24103_s23_c9953_c24290_c24098_s26_s2612_s135_s1104_s1409_s905_s24_c24202_c22143_c21373_s77_s30_U__"],["__cmpconsentx40263","BPuKNGaPuKNGaAfEHBFIABAAAAA_mABAfyA"],["consent-levels","1-1_2-1_3-0_4-0","1year"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A1%2CC0004%3A1","1year"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A0%2CC0004%3A1"],["euconsent-v2","CPuy0IAPuy0IAAHABBENDNCgAAAAAAAAAAAAJNFB_G5cSWNhOHJvY9tUaQ0HwFR4o6AgDgCZA4wBCRIAMIwF0GAAIEBAIAgAAAAEAAJAAAAEAAHAAAAAAIEBASCIAEAAIBAAICAAAAABQgAACABJGwAAEAAAAEQEABQAgAIAQBuAQEAAAAAAAAAAAAgBAABBAAAAAAAgAAAIAAAAAAgAEAAAAAAAAAAAABAAEAAAAAEAAABIaADAAEExRUAGAAIJihIAMAAQTFEQAYAAgmKMgAwABBMUdABgACCYpCADAAEExSUAGAAIJilIAMAAQTFA.YAAAAAAAAAAA"],["p","eyJnZHByX3RwIjoyLCJnZHByX3AiOjF9","1year","","reload","1"],["cmplz_consented_services","{\"youtube\":true}"],["xf_consent","%5B%22_third_party%22%5D","","","reload","1"],["je-cookieConsent","necessary","1year"],["customerCookieConsent","%5B%7B%22consentTypeId%22%3A103%2C%22consentTypeName%22%3A%22necessary%22%2C%22isAccepted%22%3Atrue%7D%2C%7B%22consentTypeId%22%3A104%2C%22consentTypeName%22%3A%22functional%22%2C%22isAccepted%22%3Atrue%7D%2C%7B%22consentTypeId%22%3A105%2C%22consentTypeName%22%3A%22analytical%22%2C%22isAccepted%22%3Afalse%7D%2C%7B%22consentTypeId%22%3A106%2C%22consentTypeName%22%3A%22personalized%22%2C%22isAccepted%22%3Afalse%7D%5D","1year"],["cookiefirst-consent","%7B%22cookiefirst%22%3Atrue%2C%22google_analytics%22%3Atrue%2C%22google_tag_manager%22%3Atrue%2C%22linkedin_ads%22%3Afalse%2C%22hubspot%22%3Atrue%2C%22twitter%22%3Afalse%2C%22active-campaign%22%3Atrue%2C%22email-marketing%22%3Atrue%2C%22bing_ads%22%3Afalse%2C%22type%22%3A%22granular%22%7D"],["consentUUID","f2fed351-3e4d-4e37-a3d6-bf942c6146e1_25"],["euconsent-v2","CP0C_wAP0C_wAAGABBENDbCgAP_AAAAAAApAJLNV_H__bX9r8X7_6ft0eY1f9_jz7uQxBhfJk-4F3LvW_JwX32E7NF36tq4KmRoEu1JBIUNlHIHUDUmwaogVrzHsakWcpTNKJ6BkkFMRU2dYCF5um4tjeQKY5_p_d3f52T-9_dv839zz38VHv3cZX--12PDdU5-9Dfn9fRfb89IL9_78v4v8_t_rk2_eT13_tetr_DAAAAaEgogAIAAXABQAFQAOQAeACAAGAAMoAaABqADwAIgATAAngBVADMAG8APQAfgBCQCGAIkARwAlgBNAClAGAAMOAZQBlgDZgHcAd8A9gD4gH2AfsA_wEAAIpARcBGACNAElAJSAUGAp4CrgFzAMUAaIA2gBuADiAIdASIAnYBQ4CjwFIgLYAXIAu8BgwDDYGRgZIAycBlwDOQGfANIgauBrIDbwG6gOCgcmBygDlwHjgPaAfSBBgCEMELQQvAhyBD0CH4EcQI-gR_AkUBJAIACAJoAtwNAEAKeAh0BnwDlBEAQAp4CHQGfAOUFQAwGfAOUAheMgBgM-AcoBC8hA4AAWABQAFwAMQAagBMACmAFUALgAYgA3gB6AEcAKUAYAAyoB3AHeAP8AigBJQCUgFBAKfAVEBVwC5gGKANoAc4A6gCVAFNAKsAWKAsoBaIC4AFyAMjAZOAzkBnwDRAGqgOAAcoA8cB9IEGAIUAQtAheBDoCHoEcQJFASQHQaQAFwAUABUADkAHwAgABdADAAMoAaABqADwAIgATAAngBVgC4ALoAYgAzABvAD0AH6AQwBEgCWAE0AKMAUoAwABhgDKAGiANkAd4A9oB9gH6AP8AikBFgEYAI6ASUAlIBQQCnwFRAVcAsQBc4C8gL0AYoA2gBuADiAHOAOoAfYBDoCLwEiAJUATIAnYBQ4CjwFNAKsAWKAsoBbAC3QFwALkAXaAu8BfQDBgGGgMegZGBkgDJwGVAMsAZcAzMBnIDPgGiANIAarA1cDWAG3gN1AcXA5MDlAHLgPHAe0A-kB9YEAQIMAQtAhfBDkEOgIegRxAjsBH0CP4EigJIDgAIC3CUDQABAACwAKAAcAA_ADAAMQAeABEACYAFUALgAYoBDAESAI4AUYAwABsgDvAH4AU-AqICrgFzAMUAdQBDoCJgEXgJEAUeAsUBZQC2IGRgZIAycBnIDPgGkANYAbeA4AB7QD6QIAgQPAgwBCECF4EPQI4gSKAkgBJUkABAW4UgpAALgAoACoAHIAPgBBADAAMYAaABqADwAIgATAAngBSACqAGIAMwAfoBDAESAKMAUoAwABlADRAGyAO-AfgB-gEWAIwAR0AkoBKQCgwFRAVcAuYBeQDFAG0ANwAdQA9oB9gEOgImAReAkQBOwChwFWALFAWwAuABcgC7QF9AMNgZGBkgDJwGWAMuAZyAz4BpEDWANZAbeA3UBwUDkwOUAcuA8UB44D2gH0gQYAhCBC0CF4EM4Icgh0BHECOwEfQI_gSKAkgUAAgLc.YAAAAAAAAAAA","1year"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A1%2CC0004%3A0%2CSTACK42%3A0","1year"],["eupubconsent-v2","CPwbUmgPwbUmgAcABBENDSCgAAAAAH_AAChQJnNf_X__b2_r-_7_f_t0eY1P9_7__-0zjhfdl-8N3f_X_L8X52M5vF36tqoKuR4ku3bBIUdlHPHcTVmw6okVryPsbk2cr7NKJ7PEmlMbM2dYGH9_n9_z-ZKY7___f__z_v-v___9____7-3f3__5__--__e_V_-9zfn9_____9vP___9v-_9_3________3_r9_7_D_-f_87_XWxBQAJMNS4gC7IkZCbaMIoEQIwrCQqgUAFEAkLRAYQurgp2VwE-sBkAIEUATwQAhgBRkACAAASAJCIAJAjgQCAQCAQAAgAVCAQAMbAAPAC0EAgAFAdCxTigCUCwgyISIhTAhKkSCgnsqEEoP1BXCEMssCKDR_xUICNZAxWBEJCxehwBICXiSQPdUb4ACEAKAUUoViKT8wBDgmbLVXgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAgAAAAA.YAAAD_gAAAAA","1year"],["CookieConsent","{necessary:true%2Cpreferences:false%2Cstatistics:true%2Cmarketing:true}","1year"],["AtomStore[COOKIES_ACCEPTED]","1","1year","","reload","1"],["AtomStore[FUNCTIONAL_COOKIES_ACCEPTED]","1","1year"],["AtomStore[MARKETING_COOKIES_ACCEPTED]","0","1year"],["solt_cookie_preferences","functional%2Cperformance","1year"],["cookie-layer-settings","{\"id\":\"b2575cbc6f69c6ff02140366ef7473e9\",\"options\":[{\"analytics-is-enabled\":false},{\"komfort-is-enabled\":false},{\"personalisierung-is-enabled\":false},{\"external-data-is-enabled\":true},{\"cf-ga-opt-in\":false},{\"cf-fb-opt-in\":false},{\"cf-go-opt-in\":false},{\"cf-sn-opt-in\":false},{\"cf-am-opt-in\":false},{\"cf-pi-opt-in\":false},{\"cf-tw-opt-in\":false},{\"cf-li-opt-in\":false},{\"cf-te-opt-in\":false},{\"cf-ti-opt-in\":false},{\"cf-ttd-opt-in\":false},{\"external-data-youtube-is-enabled\":true},{\"external-data-spotify-is-enabled\":true},{\"external-data-googlemaps-is-enabled\":true},{\"external-data-universal-is-enabled\":true}]}","1year"],["d_prefs","MjoxLGNvbnNlbnRfdmVyc2lvbjoyLHRleHRfdmVyc2lvbjoxMDAw","2629743","","domain","x.com"],["euconsent-v2","CPxurkAPxurkAAHABBENDVCgAP8AAE4AAAiQIkNf_X_fb2vj-_p99_t0eY1P9_6_t6wzjheNk-8NyZ_X_J4Xp2M6rB34pqIKuR4kunLBIQdlHGHcTUgg4IkFoSPsYk2MizJKJ7JEmlMbE2dYGG9vn8TT-ZKY70__f__zvn-r___97oAAhCAABAAAAAgAAIAAAgAIAAAAAAAAAAAAAAAAAAAAAAAADA4tAsy0bqaFsHT0Lpo4igRGjCuJWoBQGUQCwtkBhmTPCnZHAR-wnUAAxAADBByGAFEaAICCIIAkKgIkEOBAqJAIdAACgAUIBAFRIgEoiLAQCAA0B8PAKKAJSLGDKhIidcCMKxIPu-QAAEAQAAIAAQAAAABAJCgAYAAiCgGgAwABEFARABgACIKAqADAAEQUBkAGAAIgoDwAMAARBQIQAYAAiCgTAAwABEFAqABgACIKAAAA.f-AACcAAAAAA","","","domain","mundodeportivo.com"],["cookie_consent","%7B%22allowEssentials%22%3Atrue%2C%22allowFunctional%22%3Atrue%2C%22allowAnalytics%22%3Afalse%2C%22allowMarketing%22%3Afalse%7D"],["cad","$now$"],["cc","2.4"],["cp",".16.21."],["cva","W2.0"],["ubs_cookie_settings_2.0.4","0-2-1"],["ConsentChecked","{\"userHasSetCookies\":true,\"functionalityCookies\":true,\"statisticCookies\":false,\"marketingCookies\":false}","","","reload","1"],["allow_cookies","{\"essential\":\"1\",\"functional\":{\"all\":\"1\"},\"marketing\":{\"all\":\"0\"}}"],["euconsent-v2","CPywmUAPywmUAAHABBENDYCsAP_AAH_AAAAAJrNf_X__b2_r-_7_f_t0eY1P9_7__-0zjhfdF-8N3f_X_L8X52M5vF36tqoKuR4ku3bBIUdlHPHcTVmw6okVryPsbk2cr7NKJ7PkmlMbM2dYGH9_n9_z-ZKY7___f__z_v-v___9____7-3f3__5__--__e_V_-9zfn9_____9vP___9v-_9_3________3_r9_7_D_-f_87_XW-8E1ACTDQuIAuwJGQm2jCKBACMKwkKoFABRAJC0QGELq4KdlcBPrARACBFAAcEAIYAUZAAgAAAgCQiACQI4EAgEAgEAAIAFQgEABGwACgAsBAIABQHQsU4oAlAsIMiMiIUwIQpEgoJ7KhBKD9QVwhDLLACg0f8VCAgUAMVgRCQsXocASAlwkkC3VG-AAhACgFFKFYgk9MAA4JGy1B4IAA.f_gAD_gAAAAA"],["didomi_token","eyJ1c2VyX2lkIjoiMThhZDY1NzMtZGY4YS02YzJhLWJkZmUtOGI5ZjkwMDU5MmQwIiwiY3JlYXRlZCI6IjIwMjMtMDktMjdUMTE6MTI6MzcuNzk3WiIsInVwZGF0ZWQiOiIyMDIzLTA5LTI3VDExOjEyOjM3Ljc5N1oiLCJ2ZW5kb3JzIjp7ImVuYWJsZWQiOlsiZ29vZ2xlIiwiYzpkbXB3ZWJlZGktblRCSEFrNDQiLCJjOmFtYXpvbnRhbS1lWTRpTjROViIsImM6YmF0Y2gtYkp0R3R0eEwiLCJjOndhcm5lcmJyby1BUEpXeUFHUCIsImM6ZmFjZWJvb2std0RpR25KV1YiLCJjOnNuYXBjaGF0LWhXMnJNSmZZIiwiYzp0aWt0b2stV2J5cEEzWmQiLCJjOnR3aXR0ZXIteGJERXhKUGsiLCJjOmdvb2dsZWFuYS1YTXFhZ2F3YSJdfSwicHVycG9zZXMiOnsiZW5hYmxlZCI6WyJkZXZpY2VfY2hhcmFjdGVyaXN0aWNzIiwiZ2VvbG9jYXRpb25fZGF0YSJdfSwidmVuZG9yc19saSI6eyJlbmFibGVkIjpbImdvb2dsZSJdfSwidmVyc2lvbiI6Mn0="],["_iub_cs-30166201","%7B%22timestamp%22%3A%222023-09-28T08%3A20%3A53.130Z%22%2C%22version%22%3A%221.51.0%22%2C%22purposes%22%3A%7B%221%22%3Atrue%2C%222%22%3Afalse%2C%223%22%3Atrue%2C%224%22%3Afalse%2C%225%22%3Afalse%7D%2C%22id%22%3A30166201%2C%22cons%22%3A%7B%22rand%22%3A%22e747e3%22%7D%7D"],["the_green_city","[%22preference%22%2C%22technical%22]"],["cookieconsent_dismissed","1%7C%7C1%7C%7Crt2d2f69if8tq2hiq7dmjejrd2","1year","","reload","1"],["OptanonConsent","groups=C0003%3A1%2CC0002%3A1%2CC0001%3A1%2CC0005%3A1%2CC0004%3A1","1year"],["cookiepreferences","necessary#true|marketing#false"],["cookie-preferences","{\"performance\":false,\"marketing\":false,\"functionality\":true,\"social\":true,\"essential\":true}"],["hp__gdpr_consent","!demographics=false!googletagmanager=false!facebook=true!twitter=true!instagram=true!youtube=true!optimize=false!helphero=false!fbpixel=false!iterable=false!kayakWidget=false!vimeo=true!lastminute=false!apple=true!google=true!hotjar=false!getYourGuide=false!gad=false","1year","","reload","1"],["CookieConsent","{stamp:%27Uv9YOAVP5djCBw71lxhE4rNAfTObaGck2Sn3rKWf9dPCYpqcWvAlpA==%27%2Cnecessary:true%2Cpreferences:false%2Cstatistics:false%2Cmarketing:true%2Cmethod:%27explicit%27%2Cver:1}","1year"],["cm_default_preferences","{%22consent%22:[%22necessary%22%2C%22functional%22]%2C%22region%22:%22DEFAULT%22}"],["cm_eu_preferences","{%22consent%22:[%22necessary%22%2C%22functional%22]%2C%22region%22:%22EU%22}"],["gdpr","{%22version%22:%221.0_tracking%22%2C%22options%22:{%22typo3%22:true%2C%22gdpr%22:true%2C%22openstreetmap%22:true%2C%22vimeo%22:true%2C%22youtube%22:true%2C%22recaptcha%22:true%2C%22googlemaps%22:true%2C%22tracking%22:false}}"],["privacy_consent","%7B%22essentials%22%3Atrue%2C%22statistics%22%3Afalse%2C%22marketing%22%3Afalse%2C%22maps%22%3Atrue%2C%22youtube%22%3Atrue%7D"],["hs365-accepted-cookies","[{%22description%22:%22Enable%20you%20to%20navigate%20and%20use%20the%20basic%20functions%20and%20to%20store%20preferences.%22%2C%22key%22:%22technically_required%22%2C%22label%22:%22Technically%20necessary%20cookies%22%2C%22isMandatory%22:true}%2C{%22description%22:%22Display%20third%20party%20content%20like%2C%20for%20example%2C%20YouTube%20videos%20or%20twitter%20feeds.%22%2C%22key%22:%223rdParty%22%2C%22label%22:%22Third-Party%20Content%22}]"],["notion_cookie_consent","{%22id%22:%2282a41727-b652-4011-a0a4-13a4499c039a%22%2C%22permission%22:{%22necessary%22:true%2C%22preference%22:true%2C%22performance%22:false%2C%22targeting%22:false}%2C%22policy_version%22:%22v8%22}"],["eupubconsent-v2","CP0tkXAP0tkXAAcABBENAXEwAAAAAAAAAAYgAAAAAAHBAHwAAQADQAOAA8ADIALAAjgBQACuAFsARYAlwBfQDUANUAc4A8gB8gEDAIOARwAn8BQwFFALsAZwA3gB6AD5AIbAReAj0BIoCSwErAJlgTYBNoClwFPgLCgWIBYoC6gF3ALyAYFAw8DEAGLQMhAyMBkwDPwGhANGAaYA1MBroDaAG3AN0AcEA6gB2ADuoHkgeUA96B8gHyhIHAACAAFwAUABUADkAHgAgABhADIANQAeABMACqAG8APQAhIBDAESAI4ASwAmgBSgDDAGWANkAd8A9gD4gH2AfoBAICLgIwARoAlIBQQCoAFXALmAYoA2gBuADiAJEATsAocBR4CkQFNgLYAXIAu8BeYDBgGGgMkAZOAzmBrAGsgN1AcmA5cB44YCgAAEAAwADgAKAAigBOAFgAMIAeAB6AEIAI4ATAAqABXgC2ALkAcwB3AEMAIkARYAlwBSACtAGcANSAbIBwADjAHOAPIAfgBAACMAEmAJ0AUUApcBXgFfALsAX4AzgBsgDbAG8AOOAc0A6gB2QD1AHyAP2AhIBDYCLwEdQJEAkUBJcCWgJeATYAnYBQkCkQKSAU2AsUBaIC5AF0ALuAXoAwIBh4DHoGRgZIAycBnIDPAGhANMga0BrsDcgN0AdQA7gB7wgCcAAEAAwADgAKAAigBOAFgAMIAeAB6AEIAI4ATAArgBbgDmAO4AhgBEgCLAEuAKQAVoAzgBqQDZAOAAcYA5wB5AD5AH4AQAAjABJgCdAFFAKXAV4BXwC7AF-AM4AagA2wBvADjgHNAOyAeoA-QB-wEJAIbAReAjuBIgEigJLgS0BLwCbAE7AKEAUkApuBYgFigLRAXIAugBdwC9AGBAMPAY8AyQBk4DOQGhANMga0BroDdAHUAO4Ae8KARgAigBUAFgAQgAmABcAEcAKQAZQA1ABwAEcAKLAV4BXwC7AF-AM4AbwA5oB-wEegJFAS8AmwBYoC0QFsALuAXoAw8BnIDPAGhQNaA14BuQD3hgCAAEUAKgAsACEAEwARwApABqADgAI4AUWArwCvgF2AL8AZwA3gBzQD9gI9ASKAl4BNgC0QFsALuAXoAw8BnIDPAGhQNaA10B7wD4hwF6AAwAEQAOAA8AC4AGQAWAA5AB8AEcAJoAUAArgBbAC-AGgAQgAiwBHACXAFIALIAXwAwoBmgGcANQAc4A7gB5AD5gIAAgcBBwEIAI4ATiAnwCfgFFAKWAVAArIBdgC9AGcAN4AccA6QB6AD5AIbAREAioBHoCRQElgJWATEAmWBNgE2gKQAUmApcBVQCxAFlALUAWzAugC6gF3AL6AYEAxABiYDFgGQgMmAZeAz8BoQDRoGmAaaA1OBroGwANoAbcA48BywDnwHWAOwAdxA8kDygHpAPeAfEA-UB9oD8R0EcABcAFAAVAA5AB8AIAAXQAwADUAHgATAAqwBcAF0AMQAZgA3gB6gEMARIAlgBNACjAFKAMMAZQA0QBsgDvAHtAPsA_QB_wEWARgAlIBQQCrgFiALmAXkAxQBtADcAHEAOoAi8BIgCVAEyAJ2AUOAo-BTQFNgLFAWwAuABcgC7QF3gLzAYMAw0BjwDJAGTgMqAZYAzkBogDVQGsAN1AcWA5MBy4DxwH1kAFwACAAMgAsACaAF8ANAApABZAC-AGcANQAc4BHACcAE-gKGAooBSwCsgFiALSAXYA3gBzQD0AI9ASLAmwCbQFJgLEgWyBbQC7gF5AMCAYlAzwDPwGhANTAa6A2ABtwDnwHRgOwgeSB5QD3gHxEIEAACwAKAAuABiAEwAKoAXAAxABvAD0AI4Ad4A_wCUgFBAKuAXMAxQBtADqAJUAU0AsUBaIC4AFyAMnAZyA0QBqoDgAHjkgGIABgAHAAXAAyACwAHIARwAmgBUAC-AGQANoAhABSACyAGdANQA1QB3AEAAI4AT6ApoCoAFZALSAXYA3gB8gEVAI6AR6AkUBKwCWoE2ATaApMBVICxAFlALuAYsAyyBngGfgNCAa6A2ABuQDsAHlAPeJQHQAEAALAAoAByAGAAYgA8ACYAFUALgAYoBDAESAI4AUYA2QB3gD8AKuAYoA6gCLwEiAKPAWKAtgBeYDJwGcgNYAcAUgYgALgAoACoAHIAPgBBADAANQAeABMACkAFUAMQAZoBDAESAKMAUoAygBogDZAHfAPwA_QCLAEYAJSAUEAq4BcwC8gGKANoAbgBF4CRAE7AKHAU2AsUBbAC4AFyALtAXmAw0BkgDJwGcwNYA1kBuoDkwHLgPHKAQAALgAZABYADkAI4ATQAqABfADIAG0AQgAiwBHACZAFIALIAXwAwgBnQDUANUAc4A7oB8gH2AQAAjgBPgChgFLgKyArYBYoC6gLsAa8A3gB2wD0AH_AR0Aj0BIoCSwExQJsAm0BSACnwFiALoAXcAvIBfQDAgGLAMmgZ4Bn4DQgGiQNSA1OBroGwAOCAdgA7gB5QD3gHxAPlAA.YAAAAAAAAAAA","1year"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0005%3A0%2CC0004%3A0%2CC0048%3A1%2CC0030%3A1%2CC0046%3A1%2CC0032%3A1%2CC0033%3A1%2CC0057%3A1%2CC0054%3A1%2CC0056%3A1%2CC0043%3A1%2CC0029%3A1%2CC0055%3A1%2CC0026%3A1","1year"],["Vendd:AcceptedCookieConsents","FunctionalityCookies=-29&StatisticsCookies=-30&MarketingCookies=-31"],["CookieConsent","{necessary:true%2Cpreferences:true%2Cstatistics:true%2Cmarketing:true%2Cmethod:%27explicit%27%2Cver:1}","1year","","reload","1"],["_x13eucookie","{\"1\":true,\"2\":false,\"3\":false,\"4\":false,\"5\":false,\"6\":false}"],["_x13eucookie","{\"1\":true,\"2\":false,\"3\":false,\"4\":false,\"5\":true,\"6\":false}"],["_x13eucookie","{\"1\":true,\"2\":false,\"3\":false,\"5\":false,\"6\":true}"],["sj_consent_v1","%7B%22consent%22%3A%7B%22PERSONALIZATION%22%3Afalse%2C%22MARKETING%22%3Afalse%7D%2C%22uuid%22%3A%22d599dc64-52b7-4dc7-bff3-1e18d5cebc1a%22%7D"],["vl-cookie-consent-cookie-consent-date","$now$"],["consent_identifier","16f69085-c6f2-42f6-a4cd-807b24a37c0b","","","reload","1"],["consent_setting","analytics%3A0%7Cfunctional%3A1%7Cmarketing%3A0"],["zdf_cmp_client","{%22version%22:%22v1%22%2C%22iteration%22:1%2C%22consents%22:[{%22id%22:%22personalisierung%22%2C%22value%22:false}%2C{%22id%22:%22socialMedia%22%2C%22value%22:true}%2C{%22id%22:%22instagram%22%2C%22value%22:true}%2C{%22id%22:%22twitter%22%2C%22value%22:true}%2C{%22id%22:%22facebook%22%2C%22value%22:true}%2C{%22id%22:%22drittsysteme%22%2C%22value%22:true}%2C{%22id%22:%22erforderlich%22%2C%22value%22:true}%2C{%22id%22:%22erfolgsmessung%22%2C%22value%22:true}]}"],["MBCC","%7B%22version%22%3A1%2C%22functional%22%3Atrue%2C%22analytics%22%3Afalse%7D"],["ConsentStatus","%7B%22necessary%22%3Atrue%2C%22siteFeatures%22%3Atrue%2C%22statistics%22%3Afalse%2C%22marketing%22%3Afalse%7D","","","reload","1"],["CookieConsent","1.0.0.0.0"],["OptanonConsent","groups=C0003%3A0%2CC0002%3A0%2CC0004%3A1%2CC0001%3A1%2CC0009%3A0","1year"],["user","%7B%22necessary%22%3Atrue%2C%22preferences%22%3Afalse%2C%22statistics%22%3Afalse%2C%22marketing%22%3Afalse%7D"],["tarteaucitron","!tarteaucitron=true!website=true!analytics=false!regex_analytic=false!regex_ads=true!googletagmanager=false!facebookpixel=false!addtoanyshare=false!youtube=true","1year"],["cookiesSettings","%7B%22necessary%22%3Atrue%2C%22preferential%22%3Atrue%7D","","","reload","1"],["acceptedCookies","{\"necessary\":true,\"functionallity\":true,\"analysis\":false,\"marketing\":false}"],["3sat_cmp_client","{%22version%22:%22v1%22%2C%22iteration%22:1%2C%22consents%22:[{%22id%22:%22erforderlich%22%2C%22value%22:true}%2C{%22id%22:%22erfolgsmessung%22%2C%22value%22:true}%2C{%22id%22:%22personalisierung%22%2C%22value%22:false}%2C{%22id%22:%22socialMedia%22%2C%22value%22:false}%2C{%22id%22:%22twitter%22%2C%22value%22:false}%2C{%22id%22:%22instagram%22%2C%22value%22:false}%2C{%22id%22:%22facebook%22%2C%22value%22:false}%2C{%22id%22:%22drittsysteme%22%2C%22value%22:false}]}"],["cn-cc","%7B%22ccGoogleTagManager%22%3Afalse%2C%22ccGA4%22%3Afalse%2C%22ccSalesviewer%22%3Afalse%7D"],["OptanonConsent","groups=C0004%3A0%2CC0003%3A1%2CC0002%3A0%2CC0001%3A1","1year"],["OnetrustActiveGroups","C0003C0001","1year"],["userConsent","%7B%22essential%22%3Atrue%2C%22analytical%22%3Afalse%2C%22preferential%22%3Atrue%7D"],["rwc","%7B%22data%22%3A%5B%7B%22code%22%3A%22analytics%22%2C%22permissions%22%3A%5B%5D%7D%2C%7B%22code%22%3A%22marketing%22%2C%22permissions%22%3A%5B%5D%7D%5D%2C%22common%22%3A%7B%22accepted%22%3A1703366235541%7D%7D"],["OptanonConsent","groups=C0001%3A1%2CC0005%3A1%2CC0002%3A0%2CC0004%3A0%2CC0003%3A0","1year","","reload","1","domain","mesta.net"],["wmm-visitor_token","4cb8860d-4194-4ab5-be04-10f9a26afaa8"],["cookieConsent","{%22essential%22:true%2C%22tracking%22:false%2C%22marketing%22:false}"],["consts","{%22ad_storage%22:%22denied%22%2C%22analytics_storage%22:%22denied%22%2C%22functionality_storage%22:%22granted%22}"],["_CookiesPolicyAccepted","[%22necessary%22]"],["user_preferences","{\"necessary\":\"on\",\"marketing\":\"off\",\"analytics\":\"off\",\"preferences\":\"off\",\"unclassified\":\"off\"}"],["umr-data-consent","{%22essential%22:true%2C%22functional%22:null%2C%22statistics%22:false%2C%22marketing%22:false}"],["wpsaurus_cookie_banner_plugin","functional"],["_consent_cookie","{\"services\":[\"jwp\",\"twitter\"]}"],["cookies_accept","local,third"],["cookie_preference","functional"],["nero_consents","%7B%22analytics%22%3Afalse%2C%22targeting%22%3Afalse%7D"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A1%2CC0004%3A0"],["dsm-cookie-preferences","{%221%22:true%2C%222%22:true}"],["notice_gdpr_prefs","0,1,2::implied"],["notice_preferences","2:"],["cmapi_cookie_privacy","permit 1,2,3"],["qubitconsent","Declined"],["qubitconsent_version","2.0.0"],["privacyConsent_type","explicit"],["tracking-preferences","{%22destinations%22:{%22Amplitude%22:false%2C%22Braze%20Cloud%20Mode%20(Actions)%22:false%2C%22Braze%20Web%20Mode%20(Actions)%22:false%2C%22FullStory%22:false%2C%22Hindsight%22:false%2C%22Impact%20Partnership%20Cloud%22:false%2C%22Oura%20Marketing%20Id%20Middleware%20(Oura)%22:false%2C%22Statsig%22:false%2C%22Visual%20Tagger%22:false%2C%22Webhooks%22:false%2C%22Facebook%20Conversions%20API%20(Actions)%22:false%2C%22Google%20Tag%20Manager%22:false%2C%22Snap%20Conversions%20Api%22:false}%2C%22custom%22:{%22functional%22:false%2C%22marketing%22:false%2C%22advertising%22:false}}"],["SEBConsents","%7B%22version%22%3A%222%22%2C%22consents%22%3A%7B%22mandatory%22%3Atrue%2C%22statistical%22%3Afalse%2C%22marketing%22%3Afalse%2C%22simplified_login%22%3Atrue%7D%7D"],["eupubconsent-v2","CP4mi7gP4mi7gAcABBENAjEgAAAAAAAAACiQAAAAAAAA.YAAAAAAAAAAA"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A0%2CC0004%3A0%2CV2STACK42%3A0","1year"],["didomi_token","eyJ1c2VyX2lkIjoiMThkMmRlNWUtZWViOC02YmY3LWJiMGItYTA4MzI0OWNiNGFhIiwiY3JlYXRlZCI6IjIwMjQtMDEtMjFUMjE6MjE6MDAuMTg0WiIsInVwZGF0ZWQiOiIyMDI0LTAxLTIxVDIxOjIxOjA0Ljg4OFoiLCJ2ZW5kb3JzIjp7ImVuYWJsZWQiOlsiZ29vZ2xlIiwiYzpmYWNlYm9vay1ZeUpSQXllZCIsImM6eW91dHViZSIsImM6YWRvYmVhdWRpLUFXZTdyd3FkIiwiYzphbnVuY2lhbnRlX2xhX2xpZ2EiLCJjOmJlc29jeS1tRlVFYVpCTSJdfSwicHVycG9zZXMiOnsiZW5hYmxlZCI6WyJkYXRhX3NoYXJpbmciLCJkYXRhX3NoYXJpbmdfd2ViIiwiZGV2aWNlX2NoYXJhY3RlcmlzdGljcyIsImdlb2xvY2F0aW9uX2RhdGEiXX0sInZlcnNpb24iOjIsImFjIjoiREU2QU9BRVlBTmdBbFFEekFJY0FpU0RFNEFBQS5BQUFBIn0="],["euconsent-v2","CP4u7EAP4u7EAAHABBENAjEsAP_gAAAAAAAAg1NX_H__bX9r8Xr36ft0eY1f99j77uQxBhfJk-4FzLvW_JwX32EzNA26tqYKmRIEu3ZBIQFlHJHURVigaogVrzHsYkGchTNKJ6BkiFMRc2dYCF5vmYtj-QKY5_p_d3f52T-9_dv83dzzz81nv3f5f-clcLCdQ58tDfn_bRKb-5IO9_78v4v09t_rk2_eTVn_tevr7B-uft-7_XV-9_ffeAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAEAAAAxKADAAEQaikAGAAIg1DoAMAARBqIQAYAAiDUEgAwABEGotABgACINQAAA.f_wAAAAAAAAA"],["website_privacycookie_relaunch","accetta"],["website_privacycookie","accetta"],["CookieConsentLevel","functional_analytics"],["correosCookies","{\"tecnicas\":\"true\",\"analiticas\":\"false\",\"publicidad\":\"false\"}"],["bkpolicy","aceptacion=selectiva//tecnicas=si//personalizacion=si//analiticas=no//publicitarias=no//version=0","","","domain","bankinter.com"],["CookieChoicesApr2021","%7B%22functionality%22%3Afalse%2C%22anonymousMeasurement%22%3Afalse%2C%22performance%22%3Afalse%2C%22advertising%22%3Afalse%7D"],["CONSENTMGR","c1:1%7Cc2:0%7Cc3:1%7Cc4:0%7Cts:1706835520229%7Cconsent:true"],["cookie-policy-agreement","%7B%22REVISION%22%3A20%2C%22consentLevel%22%3A1%7D"],["cookiesAccepted","%7B%22necessary%22%3Atrue%2C%22statistical%22%3Afalse%2C%22marketing%22%3Afalse%7D"],["cookiepolicy","1-1722973184631"],["cookie-management-status","{\"analytics\":false,\"functional\":true,\"strictlyNecessary\":true}"],["CookieConsent","%7B%22Mode%22%3A0%7D","","","reload","1"],["aok_cookie_settings","{\"functional\":{\"key\":\"functional\",\"name\":\"Funktionale Cookies\",\"required\":true,\"approved\":true},\"comfort\":{\"key\":\"comfort\",\"name\":\"Komfort-Cookies\",\"required\":false,\"approved\":false},\"tracking\":{\"key\":\"tracking\",\"name\":\"Marketing-Cookies\",\"required\":false,\"approved\":false}}"],["user_cookie_settings","WyJwcC1mdW5jdGlvbmFsIl0=","","","reload","1"],["orejime","{\"matomo\":false,\"youtube\":true}"],["BCP","AD=0&AL=0&SM=0","1year"],["OptanonConsent","groups=1%3A1%2C2%3A0%2C3%3A1%2C4%3A0%2CV2STACK42%3A0"],["CookieConsentV2","YES%2CTOI","","","domain","matkahuolto.fi"],["CookieConsent","mandatory|external"],["consent-cookie-toggles","[{%22type%22:%22social%22%2C%22value%22:3}%2C{%22type%22:%22remaining%22%2C%22value%22:3}]"],["plenty-shop-cookie","{\"necessary\":{\"amazonPay\":true,\"consent\":true,\"consentActiveStatus\":true,\"externalId\":true,\"session\":true,\"csrf\":true,\"paypal-cookies\":true},\"tracking\":{\"googleanalytics\":false,\"bingads\":false},\"marketing\":{\"googleads\":false,\"facebookpixel\":false,\"pinterest\":false},\"media\":{\"reCaptcha\":true},\"convenience\":{\"languageDetection\":false}}"],["CONSENTMGR","c1:0|c2:0|c3:0|c4:0|c5:0|c6:0|c7:0|c8:0|c9:0|c10:0|c11:0|c12:0|c13:0|c14:0|c15:0|ts:1709021997479|consent:false|id:018de9a4f59200821a700cf146e80406f001906700ef8"],["disclaimer","2015040809"],["cookie-consent-settings","{\"necessary\":true,\"statistics\":false,\"marketing\":false,\"personalization\":true}"],["CookiesConsent","{\"ad_storage\":\"denied\",\"analytics_storage\":\"denied\",\"personalization_storage\":\"denied\"}"],["omCookieConsent","group-1.1,group-3.1,dismiss"],["ConsentCookie","required:1,functional:1,analytic:0"],["txl_consent_status","functional","","","domain","transferxl.com"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A1%2CC0004%3A0%2CC0005%3A1%2CV2STACK42%3A0","","","domain","20min.ch"],["CookiePermissionInfo","{\"LastModifiedDate\":\"1\",\"ExpirationDate\":\"1\",\"Allow\":true,\"CategoryPermission\":[{\"Category\":\"Cat.8\",\"Permission\":true},{\"Category\":\"Cat.9\",\"Permission\":true},{\"Category\":\"Cat.10\",\"Permission\":false},{\"Category\":\"Cat.12\",\"Permission\":false}]}"],["userconsent","{\"analytics\":false,\"youtube\":true}"],["aceptarCookiesSeen","{\"necesarias\":true,\"estadisticas\":false}","","","reload","1"],["cc_cookie","{\"categories\":[\"necessary\"],\"level\":[\"necessary\"],\"revision\":1,\"data\":null,\"rfc_cookie\":true,\"consent_date\":\"1\",\"consent_uuid\":\"1\",\"last_consent_update\":\"1\"}"],["UN_cookie_consent_v2","{\"marketing\":0,\"statistics\":0,\"personalization\":0,\"time\":1}","","","reload","1"],["cc_cookie","{\"level\":[\"necessary\"],\"revision\":0,\"data\":null,\"rfc_cookie\":false}"],["consentPreferences","{\"timestamp\":\"1\",\"version\":1,\"functional_storage\":\"granted\",\"analytics_storage\":\"denied\",\"personalization_storage\":\"denied\",\"ad_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_personalization\":\"denied\"}"],["cc_cookie","{\"level\":[\"tecnici\"],\"revision\":0,\"data\":null,\"rfc_cookie\":false}"],["dp_cookieconsent_status","{\"status\":\"approved\",\"checkboxes\":[{\"name\":\"statistics\",\"checked\":false},{\"name\":\"marketing\",\"checked\":false},{\"name\":\"functional\",\"checked\":true},{\"name\":\"read-speaker\",\"checked\":true}]}"],["GDPR","[\"gdpr\",\"youtube\"]"],["cookiesaccepted","{%22marketing%22:false%2C%22statistic%22:false%2C%22thirdParty%22:true}"],["sve_cc_options","{\"values\":[{\"key\":\"necessary\",\"checked\":true},{\"key\":\"ad_storage\",\"checked\":false},{\"key\":\"analytics_storage\",\"checked\":false},{\"key\":\"functionality_storage\",\"checked\":false}],\"version\":\"2023-1\",\"timestamp\":1}"],["CookieConsent","necessary:true%2C%20preference:true%2C%20statistics:false%2C%20marketing:false"],["UN_cookie_consent_v2","%7B%22marketing%22%3A0%2C%22statistics%22%3A0%2C%22personalization%22%3A0%2C%22time%22%3A1711880201%2C%22id%22%3A%221%22%7D","","","reload","1"],["cookie-consent","{\"required\":true,\"optional\":false,\"analytics\":false}"],["OptanonConsent","groups=C1%3A1%2CC2%3A0%2CC3%3A0%2CC4%3A0%2CC5%3A0%2CC8%3A0%2CC9%3A0&hosts=H184%3A1%2CH108%3A1%2CH109%3A1%2CH36%3A1%2CH58%3A1%2CH159%3A1%2CH37%3A1%2CH112%3A1%2CH331%3A1%2CH39%3A1%2CH348%3A1%2CH67%3A1%2CH115%3A1%2CH306%3A1%2CH63%3A1%2CH117%3A1%2CH118%3A1%2CH119%3A1%2CH120%3A1%2CH121%3A1%2CH122%3A1%2CH66%3A1%2CH304%3A0%2CH338%3A0%2CH40%3A0%2CH43%3A0%2CH183%3A0%2CH60%3A0%2CH274%3A0%2CH97%3A0%2CH142%3A0%2CH48%3A0%2CH49%3A0%2CH149%3A0%2CH83%3A0%2CH38%3A0%2CH176%3A0%2CH150%3A0%2CH330%3A0%2CH85%3A0%2CH151%3A0%2CH324%3A0%2CH402%3A0%2CH272%3A0%2CH14%3A0%2CH275%3A0%2CH23%3A0%2CH41%3A0%2CH101%3A0%2CH153%3A0%2CH6%3A0%2CH283%3A0%2CH42%3A0%2CH32%3A0%2CH145%3A0%2CH15%3A0%2CH322%3A0%2CH148%3A0%2CH333%3A0%2CH104%3A0%2CH55%3A0%2CH143%3A0%2CH17%3A0%2CH18%3A0%2CH57%3A0%2CH359%3A0%2CH129%3A0","1year","","reload","1"],["TC_PRIVACY","1%40032%7C1%7C4525%40%402%401713310882911%2C1713310882911%2C1747006882911%40"],["cookiesRules","{\"analytics\":false,\"personalized\":false,\"ads\":false}"],["CookieConsent","{necessary:true%2Cpreferences:false%2Cstatistics:false%2Cmarketing:true}"],["OptanonConsent","groups=BG117%3A1%2CC0001%3A1%2CC0003%3A1%2CC0002%3A1%2CSPD_BG%3A1%2CC0004%3A1"],["euconsent-v2","CQXRZAAQXRZAAD3ACQCSB6FsAP_gAEPgAATIJNQJgAFAAQAAqABkAEAAKAAZAA0ACSAEwAJwAWwAvwBhAGIAQEAggCEAEUAI4ATgAoQBxADuAIQAUgA04COgE2gKkAVkAtwBeYDGQGWAMuAf4BAcCMwEmgSrgKgAVABAADIAGgATAAxAB-AEIAI4ATgA7gCEAEWATaAqQBWQC3AF5gMsAZcBKsAA.YAAAAAAAAWAA","","","domain","seznam.cz"],["pleaseFix.privacy","{%22analyticsCookie%22:false%2C%22personalisationCookie%22:false%2C%22targetedAdvertisingCookie%22:false%2C%22multimediaPlayerCookie%22:true}"],["euconsent-v2","CP_K6QAP_K6QAAHABBENA1EsAP_gAEPgAAAAKKtV_H__bW1r8X73aft0eY1P9_j77sQxBhfJE-4FzLvW_JwXx2ExNA36tqIKmRIEu3bBIQNlHJDUTVCgaogVryDMakWcoTNKJ6BkiFMRO2dYCF5vmwtj-QKY5vr993dx2D-t_dv83dzyz4VHn3a5_2e0WJCdA58tDfv9bROb-9IPd_58v4v0_F_rE2_eT1l_tevp7D9-ct87_XW-9_fff79Ll9-goqAWYaFRAHWBISEGgYRQIAVBWEBFAgAAABIGiAgBMGBTsDAJdYSIAQAoABggBAACjIAEAAAEACEQAQAFAgAAgECgABAAgEAgAIGAAEAFgIBAACA6BCmBBAoFgAkZkRCmBCFAkEBLZUIJAECCuEIRZ4AEAiJgoAAAAACsAAQFgsDiSQEqEggS4g2gAAIAEAghAqEEnJgACBI2WoPBE2jK0gDQ04SAAAAA.f_wACHwAAAAA"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A1%2CC0003%3A1%2CC0004%3A1%2CC0005%3A1"],["pwd-cookie-consent","0,1"],["CONSENTMGR","c1:0|c2:0|c3:0|c4:0|c5:0|c6:0|c7:0|c8:0|c9:0|c10:0|c11:0|c12:0|c13:0|c14:0|c15:0|ts:1717325114359|consent:false"],["CCP","{\"publicuser\":{\"cat2\":\"off\",\"cat3\":\"off\",\"cat4\":\"off\"}}"],["mml-cookie-agreed-version","1.0.0"],["consentUUID","ce12400a-9c73-4bf8-8098-9e756366f942_34_38"],["consentUUID","798788d9-0aa9-4d39-b8f0-edb272b7ca39_32_38"],["consentUUID","6c38740a-9fcf-4ac4-8841-f3cee0293e84_34_38"],["consentUUID","4de1a7f3-4c2d-4f4e-9d18-b809b5d73ddb_33_34_38"],["consentUUID","9a37eb15-cd55-4e72-9073-e7c42c5998fd_32_38"],["consentUUID","e4ca3d8d-75f7-4b97-a468-25fc5e9dc1fd_32_33_34_38"],["consentUUID","a1751fd7-1671-4239-982b-e9ae1abe6cc3_34_38"],["consentUUID","e4fe2f39-dea0-4d85-b59e-943f634018b9_32_33_38"],["consentUUID","61ee8c80-7b19-45aa-a06d-7288610e46ed_34_38"],["consentUUID","4faab741-5ce6-430f-b701-df2242b1ff3d_34_38"],["consentUUID","a33808b0-2dce-4060-80a8-0e9d829383ef_34_38"],["consentUUID","80144095-b83e-4810-8fa7-7f9099ca6339_34_38"],["consentUUID","a673c12a-841c-4ef9-9019-1cecb12d7cdd_32_38"],["consentUUID","3c87ef3b-9471-4ad7-a677-1514eac62d74_34"],["consentUUID","8717f4cd-f101-4527-9818-21e1be78a1ef_32_38"],["consentUUID","24d18127-b8f9-4629-8d59-c8f2e5221820_34_38"],["consentUUID","8ec72931-7f30-41ee-8e7a-3eeed042376e_32_33_38"],["consentUUID","b90d74c1-472a-4b7b-b6db-6fbac1efa5af_34_38"],["consentUUID","c6ff7694-6696-4ff0-8b31-313cfe867877_34_38"],["consentUUID","b35fa5cf-67b5-4b59-a089-5e34f820f649_34"],["consentUUID","a8dac56d-c5d7-4a8a-bab2-da9eb2f28d08_34_38"],["consentUUID","de4972d9-e581-40c3-921b-296d6456fad0_34"],["consentUUID","1_34_38"],["CONSENTMGR","c1:0|c6:0|c9:0|ts:1718751098255|consent:false|id:01902d7e715a00551abb1b4878180206f003606700fb9","","","domain",".tedbaker.com"],["__Secure-HO_Cookie_Consent_Declined","1"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A1","","","domain","www.redbull.com"],["OptanonAlertBoxClosed","$currentDate$","","","domain","www.redbull.com"],["cookie_consent_202401","{\"sentry\":false,\"braze\":false,\"userpilot\":false,\"googleAnalytics\":false,\"amplitude\":false,\"hotjar\":false,\"facebookPixel\":false,\"googleAds\":false,\"kwanko\":false}"],["nmcmpv2","{\"ad\":0,\"analytics\":0,\"functionality\":0,\"personalization\":0,\"security\":0}"],["glf.functionality_consent","{\"functionality\":false,\"advertising\":false}","","","reload","1"],["wnet_vppa_assent","{\"choice\":\"decline\",\"expires\":1}"],["OptanonAlertBoxClosed","$currentISODate$"],["cc","2:f","","","reload","1"],["utwente__avgconsent","{\"a\":false,\"c\":true,\"v\":\"1\",\"lc\":\"$currentDate$\",\"isset\":true}","","","domain","utwente.nl"],["IB_PRIV_PREFS","1%7C0%7C1","","","reload","1"],["GDPR_CONSENT","ACCEPT_NECESSARY"],["klaro","{\"principal\":true,\"googleanalytics\":false,\"youtube\":false}"],["vidal_consent","tracking=0&content=0&comm=0&pub=0&social=1&date_consent=$now$&validity_date_consent=$now$","","","domain","vidal.fr"],["CONSENTMGR","c1:0%7Cc10:0%7Cc11:0%7Cc12:0%7Cc13:0%7Cc14:0%7Cc15:0%7Cc2:0%7Cc3:0%7Cc4:0%7Cc5:0%7Cc6:0%7Cc7:0%7Cc8:1%7Cc9:0%7Cconsent:true%7Cts:$now$","","","domain","1und1.de"],["tv2reg_cookie_consent","{\"revision\":1,\"consent\":[{\"handle\":\"necessary\",\"value\":true},{\"handle\":\"functional\",\"value\":false},{\"handle\":\"statistics\",\"value\":false},{\"handle\":\"preferences\",\"value\":false},{\"handle\":\"marketing\",\"value\":true}]}"],["cookieConsents","{%22required%22:true%2C%22statistics%22:null%2C%22comfort%22:true%2C%22personalization%22:null}"],["OptanonAlertBoxClosed","$currentDate$","1year","","domain","posti.fi"],["OptanonConsent","groups=C0001%3A1%2CC0003%3A1%2CC0002%3A0%2CC0004%3A0%2CC0005%3A0","1year","","domain","posti.fi"],["CookieConsent","{stamp:%27WpKgN4qzUCpEZsz6QOUxyXrv30fnb7GdP7uhCS2+tW4zpN9MSRfCig==%27%2Cnecessary:true%2Cpreferences:true%2Cstatistics:false%2Cmarketing:false%2Cmethod:%27explicit%27%2Cver:1%2Cutc:$now$}"],["cookie_tracking_enabled","4-false"],["GSK_CONSENTMGR","c1:0%7Cc2:1%7Cc3:0%7Cc4:0%7Cc5:0%7Cc6:1%7Cc7:0%7Cc8:0%7Cc9:0%7Cc10:0%7Cc11:0%7Cc12:1%7Cc13:1%7Cc14:0%7Cc15:0%7Cts:1726229678052%7Cconsent:true%7Cid:0191eb4d9db7005233a45cef34fc0406f001a06700f1c"],["boards-eu-gdpr","eu","","","domain","ignboards.com","reload","1"],["ota_ga_consent","eyJkYXRlIjoiMjAyNC0wOS0yOFQxODoyNToxMy40MjlaIiwibmVzY2Vzc2FyeSI6dHJ1ZSwicGVyZm9ybWFuY2UiOmZhbHNlLCJtYXJrZXRpbmciOmZhbHNlfQ=="],["lush-commerce-cookies","%7B%22necessary%22%3Atrue%2C%22statistical%22%3Afalse%2C%22chat%22%3Atrue%2C%22reviews%22%3Atrue%7D","","","domain","lush.com"],["cookie-preferences","{%22necessary%22:true%2C%22marketingAndAnalytics%22:false%2C%22functional%22:false%2C%22advertising%22:false}","","","reload","1","domain","app.idagio.com"],["cookie-preferences","{%22necessary%22:true%2C%22marketingAndAnalytics%22:false%2C%22functional%22:false%2C%22advertising%22:false}","","","reload","1","domain","www.idagio.com"],["ck","bbc3afa1f99650a097dc655e829ea82db262c98c%7Ee452fcf3a43bf2d79554343150020333","","","reload","1"],["cc_cookie","{\"categories\":[\"consent-log\",\"cookie-consent\",\"mapSettings\"],\"revision\":0,\"data\":null,\"consentTimestamp\":\"$currentISODate$\",\"consentId\":\"ca1ba17b-868a-47b7-82e2-fd5499b57521\",\"lastConsentTimestamp\":\"$currentISODate$\"}","1year"],["cookieConsent","%5B%7B%22name%22%3A%22essenziell%22%2C%22value%22%3A%22on%22%7D%2C%7B%22name%22%3A%22komfort%22%2C%22value%22%3A%22on%22%7D%2C%7B%22name%22%3A%22statistik%22%2C%22value%22%3A%22off%22%7D%2C%7B%22name%22%3A%22speichern%22%2C%22value%22%3A%22on%22%7D%5D"],["Cookie","accept_cookies,,","","","reload","1"],["general_cookie_consent",""],["bw_cookie_banner","{\"1\":true,\"2\":false,\"3\":false,\"timestamp\":$now$}"],["wbdLTP","68e3a742"],["luCookieConsent","{\"analyticsConsent\":false,\"marketingConsent\":false,\"expires\":\"1\"}"],["_pm_datenschutz_cookie","80ce7170aa:false;4b086eb24b:true;3a0521a8ad:false","","","reload","1"],["accept","20240307"],["preference","\"necessary,preferences\""],["consent","{\"ad_storage\":\"denied\",\"analytics_storage\":\"denied\",\"functionality_storage\":\"granted\",\"personalization_storage\":\"granted\",\"security_storage\":\"granted\"}"],["acceptCookies","{\"necessary\":true,\"preferences\":false,\"statistics\":false,\"marketing\":false}"],["PRADACONSENT_V3","c1:0|c3:0|c6:0|c9:1|ts:1732678796088|consent:true|id:01936bb4433700035e746aa2453d0406f002006700995"],["cookieAccepted","required,functional","","","reload","1"],["CookieSettingsGroupId","0193703c-779e-701d-bdba-fad9eea47938.13","","","reload","1"],["cookieConsent","{\"adConsentGranted\":false,\"adUserDataConsentGranted\":false,\"adPersonalizationConsentGranted\":false,\"analyticsConsentGranted\":false,\"personalizationConsentGranted\":false}","","","reload","1"],["cookie-consent-settings","%5B%5D"],["UBI_PRIVACY_AA_OPTOUT","true","","","domain","ubi.com"],["UBI_PRIVACY_ADS_OPTOUT","true","","","domain","ubi.com"],["UBI_PRIVACY_CUSTOMIZATION_OPTOUT","true","","","domain","ubi.com"],["UBI_PRIVACY_POLICY_ACCEPTED","true","","","domain","ubi.com"],["UBI_PRIVACY_POLICY_VIEWED","true","","","domain","ubi.com"],["tkConsentSettingsSaved","v1"],["CONSENTMGR","c1:0%7Cc3:0%7Cc9:1%7Cts:$now$%7Cconsent:true","","","domain","www.tk.de"],["cookiesConsent","{\"necessary\":true,\"statistical\":false,\"marketing\":false}","","","reload","1"],["cookieConsent","{\"essential\":true,\"analytical\":false,\"functional\":false,\"marketing\":false}"],["cookieAccepted","[\"cookieFunctionalAccepted\",null]"],["OptanonAlertBoxClosed","$currentDate$","1year","","","domain","automobiles.honda.com"],["OptanonConsent","groups=C0001:1,C0002:1,C0003:1,C0004:1","1year","","","domain","automobiles.honda.com"],["cookiepolicy","1-1861875183477"],["cookies-settings","{\"necessary\":true,\"functional\":false,\"statistical\":false,\"marketing\":false}"],["cookie_consent_v1","reject_all"],["pref","%7B%22t%22%3A%22explicit%22%2C%22f%22%3Afalse%2C%22a%22%3Afalse%2C%22m%22%3Afalse%7D"],["cookie-perms","2%3A$now$%7Cf","","","domain","stripe.com"],["daveConsentChoices","{\"analytics_storage\":\"denied\",\"personalization_storage\":\"denied\",\"ad_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_personalization\":\"denied\",\"functionality_storage\":\"granted\",\"security_storage\":\"granted\"}"],["consent_cookie","functional%2Cnecessary","1year","","reload","1"],["_pandectes_gdpr","eyJzdGF0dXMiOiJjdXN0b20iLCJ0aW1lc3RhbXAiOjE3MzQ3OTQ2MjQsInByZWZlcmVuY2VzIjo2LCJpZCI6IjY3NjZkZDgxY2VmNzg3NzE0ZjU4YzAxZSJ9","1year"],["pons-pure","consent-given","","","domain",".pons.com"],["FMI_COOKIE_CONSENT","%7B%22ts%22%3A%221970-01-01T00%3A00%3A00.000Z%22%2C%22allow%22%3A%7B%22necessary%22%3A%7B%22fmi_necessary%22%3A1%7D%2C%22preferences%22%3A%7B%22fmi_preferences%22%3A1%7D%2C%22analytics%22%3A%7B%22rs%22%3A1%2C%22userneeds%22%3A2%7D%2C%22other%22%3A%7B%22youtube%22%3A1%2C%22fyrebox%22%3A1%7D%7D%7D","1year"],["CookieConsentResult","$currentDate$"],["usercentrics_consent","{\"ad_storage\":0,\"analytics_storage\":0,\"functional\":0}"],["je-cookieConsent","necessary","","","domain","lieferando.de"],["customerCookieConsent","[{\"consentTypeId\":103,\"consentTypeName\":\"necessary\",\"isAccepted\":true,\"decisionAt\":\"$now$\"},{\"consentTypeId\":104,\"consentTypeName\":\"functional\",\"isAccepted\":true,\"decisionAt\":\"$now$\"},{\"consentTypeId\":105,\"consentTypeName\":\"analytical\",\"isAccepted\":false,\"decisionAt\":\"$now$\"},{\"consentTypeId\":106,\"consentTypeName\":\"personalized\",\"isAccepted\":false,\"decisionAt\":\"$now$\"}]","","","domain","lieferando.de"],["consentCookieV2","{\"consentAnalytics\":false,\"consentAds\":false,\"functionality_storage\":true,\"security_storage\":true,\"personalization_storage\":true,\"ad_storage\":false,\"ad_user_data\":false,\"ad_personalization\":false,\"analytics_storage\":false}","15778476"],["CookieConsentResult","$currentISODate$"],["cookie-optIn","{\"required\":true,\"statistics\":false,\"thirdParty\":true}"],["cookies-consent|kanga","1836378181|NECESSARY"],["cpc_en_ww","eyJhIjoxLCJtIjoxLCJ2IjoxLjl9"],["_consentCookie","{\"categories\":[\"required\"]}"],["CookieConsentSyreta","eyJ2ZXJzaW9uIjoxLCJjYXRlZ29yeSI6WyIxIl19"],["cookie_consent","{\"cookies-necessary\":true,\"cookies-preferences\":true,\"cookies-analytical\":false,\"cookies-marketing\":false,\"cookies-other\":false}"],["cookie_consent_status","[\"necessary\"]","","","reload","1"],["cc_cookie","{\"categories\":[\"necessary\"],\"level\":[\"necessary\"],\"revision\":0,\"data\":null,\"rfc_cookie\":false,\"consent_date\":\"$currentISODate$\",\"consent_uuid\":\"a3cbbb35-63e1-4ee9-ab8c-b327fd3a863e\",\"last_consent_update\":\"$currentISODate$\"}"],["cookies_consent_disclaimer","closed","","","reload","1"],["VersionedCookieConsent","v:2,essential:1,analytics:0,social:0,perso_cont_ads:0,ads_external:0"],["AcceptedCookies","[\"essential\"]"],["errCookieConsent","necessary,functional,embed"],["cookie_policy","1718029577"],["cookie_consent","{\"functional\":true,\"analytics\":false}"],["cookie_agreement_version","cookie-agreement-V1","","","reload","1"],["cpc_en_gb","eyJhIjowLCJtIjowLCJ2IjoxLjl9"],["privacy","{\"marketing\":false,\"analytics\":false}"],["consent_date","$now$"],["consent_settings","{\"consentAnalytics\":false,\"consentMarketing\":false\"lastUpdated\":1}"],["cookiesjsr","%7B%22functional%22:false,%22gtag%22:false,%22recaptcha%22:false,%22addtoany%22:false,%22twitter%22:true,%22video%22:true,%22vimeo%22:true%7D"],["cookieConsent","{\"ad_storage\":\"denied\",\"analytics_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_personalization\":\"denied\"}"],["orejime","{\"functional\":true,\"analytical\":false,\"marketing\":false,\"targeting\":false,\"analytical_and_marketing_and_targeting\":false}"],["tf_respondent_cc","{\"groups\":[],\"timestamp\":\"1\"}"],["ConsentCookie","required:1,functional:1,analytic:0","","","reload","1"],["controlCookies","{\"functional\":true,\"analytics\":false,\"publicity\":false}"],["sj_consent_v1","%7B%22consent%22%3A%7B%22PERSONALIZATION%22%3Afalse%2C%22MARKETING%22%3Afalse%7D%2C%22uuid%22%3A%221%22%7D"],["_nasStorageConsent","all=False&analysis=False&marketing=False&v=1"],["klett_cookie_consent","{%22statistiken%22:false}"],["rsdperms","%7B%22ringside%22%3Afalse%2C%22googleads%22%3Afalse%2C%22facebook%22%3Afalse%2C%22read%22%3Atrue%7D"],["muziker_consent","%7B%22marketingCookie%22%3Afalse%2C%22analyticsCookie%22%3Afalse%7D","","","reload","1"],["cookie_preferences","{\"allow_all\":0,\"google_analytics\":0,\"accepted\":1}","","","reload","1"],["noticeCookie","{\"cookieNeeded\":true,\"cookiePerfomance\":false,\"cookieFunctionality\":false,\"cookieTargeted\":false,\"cookieSocial\":false}"],["tavex_cookie_consent","{\"necessary\":true,\"preference\":false,\"marketing\":false,\"statistic\":false}"],["OESP_LGPD_ACEITE","VERDADEIRO"],["cookiesPolicy","{\"analytics\":false,\"marketing\":false,\"functional\":false,\"essential\":true}"],["cookie-agreed-version","1.0.0"],["consent","%7B%22analytics%22%3Afalse%2C%22marketing%22%3Afalse%2C%22functional%22%3Atrue%7D"],["cookie_consents","{\"essential\":true,\"functional\":false,\"analytical\":false,\"promotional\":false}","","","reload","1"],["ck_consent","AT","","","reload","1"],["cookie_preferences","{\"allow\":[]}"],["cookiesSettings","{\"necessary\":true,\"preferential\":false,\"analytics\":false,\"marketing\":false}","","","reload","1"],["CONSENTMGR","c2:0%7Cc3:0%7Cts:1737578911415%7Cconsent:false"],["ett_cookie_acceptance","{\"strictlyNecessaryCookies\":true,\"analysisCookies\":false,\"functionaliyOrPersonalizationCookies\":false}"],["pr-cookie-consent","[]"],["maandag-cookie-consent","{\"functional\":true,\"analytics\":false,\"advertising\":false,\"personalisation\":false,\"maccepted\":1}","","","reload","1"],["cookiePolicyGDPR__details","{\"cookieCreateTimestamp\":$now$}"],["cs-cookie-banner","|essential|"],["tlnk_cwc","necessary_cookies=true"],["nlportal.cookieConsent","basis"],["_uc","ad_storage=denied:analytics_storage=denied"],["noticeCookie","{\"cookieNeeded\":true,\"cookiePerfomance\":true,\"cookieFunctionality\":false,\"cookieTargeted\":false,\"cookieSocial\":false}"],["polityka-prywatnosci-b4utxgfk","on_init","","","reload","1"],["privacy_settings","{\"analytics_storage\":\"denied\",\"ad_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_personalization\":\"denied\",\"personalization_storage\":\"denied\"}"],["ty-cookies","{\"necessary\":true,\"performance\":false,\"functional\":false,\"advertising\":false,\"visitor_guid\":\"1\"}"],["gdpr_preferences","{\"consent\":false,\"pref\":{\"analytics\":0,\"targeting\":0,\"thirdparty\":0},\"ts\":1}"],["CookieConsent","mandatory"],["LeetchiRGPDCookie","{\"security\":true,\"analytics\":false,\"marketing\":false}"],["analyticsConsents","{\"necessary\":true,\"analytics\":false,\"marketing\":false}"],["siwa-cb","media=true,essentials=true"],["cookie_consent","agree_none","","","reload","1"],["cookie-concent","{\"marketing\":false,\"statistics\":false}"],["cookie_consent","\"analytics=-1\""],["cookie_preferences","[{\"type\":\"essential\",\"status\":\"on\"},{\"type\":\"statistics\",\"status\":\"off\"},{\"type\":\"performance\",\"status\":\"off\"},{\"type\":\"marketing\",\"status\":\"off\"}]","","","reload","1"],["cookies_confirmed","1;2,3,4,5","","","reload","1"],["cookie_time_prod","$now$"],["_cookie-consent","{\"version\":\"1\",\"consent\":\"yes\",\"categories\":[]}"],["cookie_consent","{\"required_cookies\":\"true\",\"permanent_cookies\":\"false\",\"statistics_cookies\":\"false\",\"marketing_cookies\":\"false\"}","","","reload","1"],["rodoCookieSettings","[{\"purpose\":\"required\",\"value\":true},{\"purpose\":\"youtube\",\"value\":true},{\"purpose\":\"ga\",\"value\":false},{\"purpose\":\"fbpixel\",\"value\":false},{\"purpose\":\"gads\",\"value\":false},{\"purpose\":\"matomo\",\"value\":false}]"],["cookiesConsent","{\"technical\":true,\"personalization\":false,\"analytics\":false}"],["gCookieConsent","{\"stats\":false,\"prefs\":false,\"ads\":false,\"updatedAt\":1,\"respondedAt\":1}"],["_cfy_cc","%7B%22Consents%22%3A%7B%22essential%22%3Atrue%2C%22analytics%22%3Afalse%2C%22marketing%22%3Afalse%7D%7D","","","reload","1"],["consent-cookie-storage","{\"cookies-checkbox-essential\":true,\"cookies-checkbox-statistics\":false,\"cookies-checkbox-external-media\":true,\"cookies-checkbox-googlemaps\":true}"],["cookieConsent","{\"essential\":1,\"statistics\":0,\"advertisement\":0}"],["cookieConsent","functional=1&analytics=0&marketing=0&updated=1","","","reload","1"],["cookie_preference","[{\"key\":\"necessary\",\"value\":true},{\"key\":\"analytics\",\"value\":false},{\"key\":\"personalization\",\"value\":false},{\"key\":\"third-party\",\"value\":true}]"],["cookie_accepted","{\"first_option\":true,\"second_option\":false}"],["cookiejs_preferences","{\"preferences_selected\":true}"],["consent","{\"functionality\":true,\"security\":true,\"ad\":false,\"personalization\":false,\"analytics\":false}","","","reload","1"],["consent","false"],["cookieprefs","{\"func\":false,\"perf\":false,\"trgt\":false,\"thrd\":false}"],["cookies_consent","!gtag=false"],["bm-cookies-settings-v2","{\"basic\":true,\"functional\":false,\"analytic\":false}"],["anthropic-consent-preferences","{\"analytics\":false,\"marketing\":false}","","","reload","1"],["BIH-Cookie","requiredCookies"],["COOKIE_CONSENT","ALLOW_REQUIRED","","","reload","1"],["gdprconsent","{basic:true,useful:false,analytics:false,marketing:false}"],["cookiesConsent","{\"marketing\":false,\"analytic\":false,\"preferences\":false}","","","reload","1"],["cookiesjsr","{\"base\":false,\"analytics\":false}"],["cookiesjsr","{\"functional\":true,\"recaptcha\":true,\"video\":true}"],["cookie_flag","{\"necessary\":1,\"analytics\":0}"],["COOKIES_SETTINGS_N","technicalallowed"],["isUseCookie","vietnamairlines_usingCookie"],["cookie-processed-02","ck_1:true%2Cck_2:false"],["cookieConsent","{\"functional\":true,\"security\":true,\"analytics\":true,\"marketing\":false,\"personalization\":false,\"ad_personalization\":false,\"ad_user\":false}"],["cookie-consent","{\"functional\":false,\"marketing\":false}"],["cookie_consent","{\"technical\":true,\"marketing\":false,\"statistical\":false,\"cjValue\":null}","","","reload","1"],["cookie-policy-acceptance","{\"system\":true,\"default\":false,\"publicity\":false,\"analyse\":false}"],["cookie-cart-summary-v4","{\"q\":0}"],["klaro","{\"default\":true,\"privacy\":true,\"youtube\":true,\"vimeo\":true,\"trusted_shops\":false,\"google_analytics\":false,\"facebook\":false,\"hotjar\":false,\"awin\":false,\"ads\":false,\"sentry\":false}"],["allowCookies","{\"_ga\":false,\"_gat\":false,\"_gid\":false,\"_ga_MNGQPQ0S9L\":false,\"uvc\":false,\"__cfduid\":false}"],["cookie-agreed-categories","[\"necesare_de_functionalitate\"]"],["cookieconsent_status","!tarteaucitron=true!website=true!youtube=true!openstreetmap=true!matomo=false"],["consent_cookiespolicy","V2"],["consent_privacypolicy","V2"],["consent_termsofuse","V3"],["_consentik_cookie","[{\"categoriesSelected\":[]}]"],["evendo_privacysettings","{\"confirmed\":1,\"level\":2,\"state\":{\"t\":{\"accept\":1},\"p\":{\"accept\":0},\"s\":{\"accept\":0},\"k\":{\"accept\":0}}}"],["_x13eucookie","{\"1\":true,\"2\":false,\"3\":false,\"4\":false,\"5\":true}"],["elgrafico_com_ar","{\"consent\":\"ca\"}"],["cookiesjsr","{\"functional\":false,\"analytics\":false,\"comscore\":false,\"video\":true,\"cookies_publicitarias\":false}"],["gen_consent","{\"choices\":{\"site-improve\":false,\"google-analytics\":false,\"google\":false,\"watchity\":false,\"twitter\":false}}"],["allowedCookies","technical"],["cookie_consent_consent","gived"],["melindres_options","{\"required\":true,\"analytics\":false,\"advertising\":false}"],["_consentik_cookie","[{\"categoriesSelected\":[\"necessary\"]}]"],["dp_cookieconsent_status","{\"status\":\"approved\",\"checkboxes\":[{\"name\":\"statistics\",\"checked\":false},{\"name\":\"marketing\",\"checked\":false},{\"name\":\"googlemaps\",\"checked\":false},{\"name\":\"require\",\"checked\":true}]}"],["waconcookiemanagement","setwcmmints1"],["air_cookie_consent","{\"1\":true,\"2\":false,\"3\":false,\"4\":false}"],["user_cookie_consent","{\"analytics\":false,\"ad_storage\":false}"],["cookie-ag","{\"necessary\":1,\"functional\":0,\"performance\":0,\"marketing\":0}"],["consent_cookie","{\"ad_storage\":\"denied\",\"analytics_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_personalization\":\"denied\"}","","","reload","1"],["cookieBasic","$now$"],["nebo","{\"acceptance\":true,\"analytics\":false}"],["CookieChoicesApr2021","{\"functionality\":false,\"anonymousMeasurement\":false,\"performance\":false,\"advertising\":false}"],["cookie_consent","{\"rejectAll\":true}"],["cookie-consent","given"],["cb-enabled","declined"],["cookie_consent","{\"consent\":true,\"options\":[\"calendly\",\"google_maps\",\"podigee\",\"vimeo\",\"youtube\"]}"],["gibgas","ga1%3D0"],["dw_cookies_accepted","{\"other\":false,\"analytical\":false,\"advertising\":false,\"social_networks\":false}"],["cookie_policy_version","1.0.1"],["EtherFiGDPR","{\"functionalEnabled\":true,\"analyticsEnabled\":false}"],["cookie-agreed-version","2.0"],["cookie-agreed-categories","[\"necessary\",\"functional\"]"],["cds_cs","{\"cookies\":{\"cookies_performance_analytics\":0,\"cookies_functional\":0,\"cookies_marketing_targeting\":0},\"version\":6,\"all\":0}"],["CookieConsentResult","$now$"],["cookieConsent","{\"necessary\":\"1\",\"statistics\":\"0\",\"advertising\":\"0\"}","","","reload","1"],["GDPR_Consent","{\"essential\":true,\"functional\":true,\"analytical\":false,\"advertising\":false}"],["consent_setting","analytics:0|functional:1|marketing:0","","","reload","1"],["COOKIE_CONSENT","{\"technical-cookies\":true,\"analytics\":false,\"ads\":false}"],["iabbb_cookies_policy","%7B%22necessary%22%3Atrue%2C%22functional%22%3Afalse%2C%22performance%22%3Afalse%2C%22marketing%22%3Afalse%7D","","","domain",".bbb.org"],["cookie_setting","{\"functional\":true,\"analytic\":false}"],["userPrivacyCookie","{\"analytics\":{\"enabled\":false,\"date\":1},\"marketing\":{\"enabled\":false,\"date\":1}}"],["cookie-agreed-categories","[\"mandatory\"]"],["allow-cookies","essential-only"],["cookieControlPrefs","[\"essential\"]"],["cb_permissions","[\"Notwendige\"]"],["gdpr2","base"],["cookieControlPrefs","[]"],["cookieControlPrefs","[\"essential\",\"medien\"]"],["VisitorCookieSettings","{\"functional\":true,\"advertising\":false,\"tracking\":false,\"timestamp\":1}"],["cookieConsent","{\"necessary\":true,\"analytics\":false,\"personalize\":false,\"marketing\":false}"],["COOKIE_CONSENT","NECESSARY|NECESSARY.DSGVO_CONSENT|TECHNICAL|TECHNICAL.PHP_SESSION|TECHNICAL.SHOP_WARENKORB|TECHNICAL.SHOP_PAYMENT|TECHNICAL.NEWSLETTER"],["CookieInformationConsent","{\"website_uuid\":\"0fabd588-2344-49cd-9abb-ce5ffad2757e\",\"timestamp\":\"$currentDate$\",\"consent_url\":\"https://www.karkkainen.com/\",\"consent_website\":\"karkkainen.com\",\"consent_domain\":\"www.karkkainen.com\",\"user_uid\":\"\",\"consents_approved\":[\"cookie_cat_necessary\",\"cookie_cat_functional\"],\"consents_denied\":[],\"user_agent\":\"\"}"],["cookiesSettings","{\"necessary\":true}","","reload","1"],["Consent_status","!youtube=true!matomo=false!tarteaucitron=true"],["COOKIE_OPT_IN","NO_OPT_IN"],["consents","{\"marketing\":false,\"marketing.marketing.marketing\":false,\"necessary\":true,\"necessary.necessary.necessary\":true,\"preferences\":false,\"preferences.preferences.preferences\":false,\"statistics\":false,\"statistics.statistics.statistics\":false}"],["cookieConsent","{\"preferences\":false,\"analytics\":false,\"marketing\":false}"],["hide_cn","core"],["cookiesOptions","{\"ad_storage\":0,\"analytics_storage\":0}"],["fc_cookieconsent_status","general"],["orejime","{\"site\":true,\"analytics\":false}"],["cookieControlPrefs","[\"maps\"]"],["cookieConsent","{}"],["privacy","{\"cga\":0,\"cgc\":0,\"cfb\":0,\"cfb_share\":0,\"cvwo\":0}"],["cookie_settings","{\"accepted\":true,\"performance\":false,\"functional\":false}"],["dragnsurvey_cc","{\"choice\":true,\"analytics\":false}"],["CookiePreferences","[]"],["cookies-agreed","{\"essential\":true,\"ad_storage\":false,\"analytics_storage\":false,\"ad_user_data\":false,\"ad_personalization\":false,\"version\":2}"],["cookieconsent_mode","[]"],["orejime","{\"google-analitics\":false,\"wordpress\":true}"],["orejime","{\"strictly_necessary\":1,\"tracking\":false}"],["orejime","{\"strictly_necessary\":1,\"iframe_youtube\":false,\"iframe_vimeo\":false}"],["orejime","{\"core\":true,\"analytics\":false,\"streaming\":false}"],["cookiebox","cookies-essentiels,cookiebox"],["fs-cc","%7B%22id%22%3A%22CtPFSrvSu0U7lxOz8lYFN%22%2C%22consents%22%3A%7B%22analytics%22%3Afalse%2C%22essential%22%3Atrue%2C%22marketing%22%3Afalse%2C%22personalization%22%3Afalse%2C%22uncategorized%22%3Afalse%7D%7D"],["configuracion_cookies","true|false"],["_notified_","Cookies verified:"],["klaro","{\"functional\":true,\"analytics\":false,\"marketing\":false,\"others\":false}"],["_pCookies","[\"2\",\"3\",\"1\"]"],["cconsent","{\"version\":1,\"categories\":{\"necessary\":{\"wanted\":true},\"Microsoft\":{\"wanted\":false},\"Google\":{\"wanted\":false},\"Facebook\":{\"wanted\":false},\"YouTube\":{\"wanted\":false}},\"services\":[\"Microsoft\",\"analytics\",\"facebook\",\"YouTube\"]}"],["consent-policy","{\"ess\":1,\"func\":0,\"anl\":0,\"adv\":0,\"dt3\":0}"],["gox_cookie_consent","websiteTranslator"],["complianceCookie","{\"necessary\":true,\"statistics\":false,\"marketing\":false}"],["gdprCookiesAccepted","bhWebsiteAccepted:true"],["vzs_consent_necessary","granted"],["razzi_newsletter_popup_prevent","$now$"],["cookietype","strictly_necessary"],["museumnl-cookie-consent","{\"generalConsent\":true,\"version\":3}"],["RP_PRIVACY_SETTINGS","{\"isCookiesAccepted\":true}"],["bp_cookiepopup","{\"performance\":0,\"targeting\":0}"],["cookie_consent","\"preferences=-1\"","","","reload","1"],["cookieConsent","{\"necessary\":true,\"basic\":true,\"experience\":false,\"measurement\":false,\"targeting\":false}"],["brd_cookie_consent","{\"name\":\"accept_mkt\",\"saved_at\":1,\"settings\":{\"necessary\":true,\"pers\":false,\"perf\":false,\"mkt\":false}}"],["complianceCookie","system"],["CookieScriptConsent","{\"action\":\"reject\",\"categories\":\"[]\"}"],["gdpr-opt-ins","[]"],["gdpr-categories","[%22required%22]"],["tarteaucitron","!googletagmanager=false!matomotm=false!addtoanyshare=false!vimeo=false!youtube=false"],["csrv_cookie_consent","{\"essential\":true,\"ad_storage\":false,\"analytics_storage\":false,\"ad_user_data\":false,\"ad_personalization\":false}"],["ols_cookieconsent","disagree"],["cookie-agreed-categories","[\"technical_cookies\"]"],["cookienote","set"],["accepted_optional_cookies","statistics:0|marketing:0","","","reload","1"],["BokioDNT","DoNotTrack"],["skovborghoghCookieConsent","{\"functionalOnOff\":false,\"statisticalOnOff\":false,\"marketingOnOff\":false,\"cookieId\":\"1\",\"dateOfConsent\":\"1\"}"],["lz-us-ac","NE%2CAN%2CMK","","","reload","1"],["cookieInfo","ja"],["dataMercedesBenz","Marketing"],["apcAcceptedTrackingCookie3","10000001"],["tmp_eu_cookie_consent","{\"strictly_cookie\":\"1\"}","","","reload","1"],["COOKIE_CONSENT","functi-0|market-0"],["__w_cc","{\"preferences\":false,\"analysis\":false,\"marketing\":false}"],["GoogleAnalyticsUserConsentCookie","{\"ad_storage\":\"denied\",\"analytics_storage\":\"denied\",\"functionality_storage\":\"denied\",\"personalization_storage\":\"denied\",\"security_storage\":\"denied\"}"],["SitecoreUserConsentCookie","NoTracking"],["toc_cookie_consent","strictly_necessary"],["cookiefirst-consent","%7B%22necessary%22%3Atrue%2C%22performance%22%3Afalse%2C%22functional%22%3Atrue%2C%22advertising%22%3Afalse%2C%22type%22%3A%22category%22%7D","1year"],["gnb-cookie-agree","agree"],["gdpr","mandatory=1,marketing=0,media=1"],["cookiesjsr","{\"what_are_cookies_\":true,\"analytical_cookies\":false,\"functional\":true,\"functional_cookies\":true}"],["allowCookies","[]"],["cookiebar","marketing=0&statistic=0","","","reload","1"],["CookieConsent","{\"timestamp\":1,\"marketing\":false,\"personalization\":false,\"statistics\":false}"],["COOKIES","1|"],["neoCookieConsent","{\"functionality_storage\":\"granted\",\"analytics_storage\":\"denied\",\"ad_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_personalization\":\"denied\",\"personalization_storage\":\"denied\",\"security_storage\":\"denied\"}"],["cookies-preferences","{\"functionalCookies\":true,\"analyticalCookies\":false,\"marketingCookies\":false}"],["cookiebar","{\"analytics_storage\":\"denied\",\"ad_storage\":\"denied\",\"personalization_storage\":\"denied\"}"],["ppw-cookie","{\"consents\":[12],\"uuid\":\"1\",\"acceptance-time\":\"1\"}"],["orejime","{\"system\":false,\"google-analytics\":false}"],["cookieConsent","basis"],["thbcookies","{\"1\":0}"],["CMP","{\"categoryPreferences\":{\"essential\":true,\"analytics\":false,\"functional\":false,\"advertising\":false}}","","","reload","1"],["user_allowed_save_cookie_v2","%7B%224%22%3A1%7D","1year"],["user_allowed_save_cookie","%7B%224%22%3A1%7D","1year"],["cookie-consent-1","{\"required_storage\":true,\"functionality_storage\":true,\"statistics_storage\":false,\"ad_storage\":false}"],["cookie-consent-2","%7B%22version%22%3A2%2C%22categories%22%3A%7B%22required%22%3Atrue%2C%22improved_experience%22%3Afalse%2C%22partners%22%3Afalse%7D%7D","1year"],["popup-cookie-alertca","viewed"],["cookie_consent","necessary=granted&marketing=denied&functional=denied"],["CookieConsent","{necessary:true%2Cpreferences:true%2Cstatistics:false%2Cmarketing:false%2Cmethod:%27explicit%27}","1year"],["pow-cookie-cats-34421","[]","","","reload","1"],["OptanonConsent","groups=0:1","1:0","2:0","3:0","1year"],["cookie-consent","{\"functional\":true,\"analytics\":false,\"marketing\":false}","","","reload","1"],["_deCookiesConsent","[%22needed%22]"],["gdpr","{\"types\":[\"ESSENTIAL\"],\"platform\":\"web\"}"],["cookies_and_content_security_policy","[]"],["cookieConsentPrefs","{\"cg_essential\":1,\"cg_analytical\":0,\"cg_marketing\":0,\"expire\":1,\"secure\":1}"],["RBXcb","RBXViralAcquisition=false&RBXSource=false&GoogleAnalytics=false"],["fyCcDecision","{\"ccStatistics\":false}"],["gdpr_c","c","","","reload","1"],["CookieScriptConsent","{\"action\":\"reject\",\"consenttime\":1,\"categories\":\"[]\"}"],["gdprcookienotice","{\"date\":\"1\",\"necessary\":true,\"performance\":false,\"analytics\":false,\"marketing\":false}"],["privacy-preferences","{\"essential\":true,\"analytics\":false}"],["userconsent","v2|"],["in2cookiemodal-selection","[%22essential%22]"],["STORAGE_PREFERENCES","{\"STRICTLY_NECESSARY\":true,\"PERFORMANCE\":false,\"FUNCTIONAL\":false,\"TARGETING\":false,\"SOCIAL_MEDIA\":false,\"PIXEL\":false,\"__VERSION\":4}"],["moove_gdpr_popup","%7B%22strict%22%3A%221%22%2C%22thirdparty%22%3A%220%22%2C%22advanced%22%3A%221%22%7D"],["OptanonConsentInSided","C0002"],["tarteaucitron","!affilae=false!googleadwordsconversion=false!googleadwordsremarketing=false!plausible=false!googletagmanager=false!recaptcha=true"],["tarteaucitron","!mc_analytics=false!siteimproveanalytics=false!facebook=false!instagramembed=false!twitterembed=false!youtube=false"],["tarteaucitron","!gajs=false!googletagmanager=false!facebook=false!twitter=false!dailymotion=false!vimeo=false!youtube=false"],["okosgazdi_gdpr_cookies_accepted_list","{\"0\":\"necessary\"}"],["sisu_cookie_consent","[\"necessary\"]","","","reload","1"],["cookiebanner.services.accepted","accessibility"],["cookiebanner.version","0.0.3"],["cookie-agreed-categories","[%22required_cookies%22]"],["mycookies_v2","{\"statistik\":false}"],["cc_cookie_accept","null"],["cc_cookie_decline","cc_cookie_decline"],["cookie-consent","{\"ad_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_personalization\":\"denied\",\"analytics_storage\":\"denied\",\"functionality_storage\":\"denied\",\"personalization_storage\":\"denied\",\"security_storage\":\"granted\"}"],["ktrack","{\"fonctionnels\":true,\"publicitaires\":false,\"statistiques\":false,\"tiers\":false}"],["cookie-agreed","C0000"],["cookieConsentGroups","{\"adConsentGranted\":false,\"adPersonalizationConsentGranted\":false,\"adUserDataConsentGranted\":false,\"analyticsConsentGranted\":false,\"functionalityConsentGranted\":true,\"personalizationConsentGranted\":false,\"securityConsentGranted\":true}"],["ac","{\"version\":\"1\",\"functional_storage\":\"denied\",\"analytics_storage\":\"denied\",\"ad_storage\":\"denied\"}","","","reload","1"],["hw-cc-first-party","{\"uuid\":\"1\",\"statistics\":false,\"marketing\":false}"],["hw-cc-gtm","{\"statistics\":true,\"marketing\":false}"],["Cetin Cookies Consent","{\"ad_storage\":\"denied\",\"analytics_storage\":\"denied\",\"functionality_storage\":\"granted\",\"personalization_storage\":\"denied\",\"security_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_personalization\":\"denied\"}"],["tarteaucitron","!gtag=false!youtube=true"],["cookieChoices","{\"search\":\"disable\",\"weight\":\"disable\",\"geoloc\":\"enable\"}"],["consent","false_false_true"],["Cookie_Consent","1%2C1%2C1%2C0%2C0"],["consentCookie","{\"necessary\":\"NO_CHOICE\",\"functional\":\"DENIED\",\"performance\":\"DENIED\",\"targeting\":\"DENIED\",\"social\":\"DENIED\"}"],["cookie_consent","{\"essential\":true,\"uid\":\"1\"}","","","reload","1"],["cookieSettings","1,0,0,0"],["__cc_prefs","{\"saved\":true,\"events\":{\"cc_necessary_given\":0,\"cc_tracking_given\":0},\"date_accepted\":\"1\"}"],["cookie-agreed-categories","[\"essential\"]"],["laravel_eu_cookie_consent","{\"essential_cookies\":\"1\"}","","","reload","1"],["aCookieSel","{\"formParams\":null,\"action\":\"saverequired\"}"],["ongeo_cookie","{\"cookies-necessary\":true,\"cookies-preferences\":false,\"cookies-analytical\":false,\"cookies-marketing\":false,\"cookies-other\":false}"],["cookieConsent","{\"submitBasicData\":true,\"submitUsageStats\":false,\"submitAudienceData\":false}","","","reload","1"],[".consent","fu0-ma0-pe0","","","reload","1"],["cookiehub","eyJhbnN3ZXJlZCI6dHJ1ZSwicmV2aXNpb24iOjEsImRudCI6ZmFsc2UsImFsbG93U2FsZSI6dHJ1ZSwiaW1wbGljdCI6ZmFsc2UsInJlZ2lvbiI6IiIsInRva2VuIjoiRGxJaWVvNzVCWmNubFQ1eHlPRmNlaUV4cjZnWVROajhjQVd6OWdVQW9IT1pGSzhyNGlyMDZiYjhTNkNuOWc0bCIsInRpbWVzdGFtcCI6IjIwMjUtMDItMjNUMjA6NTc6MzkuMzYyWiIsImFsbEFsbG93ZWQiOmZhbHNlLCJjYXRlZ29yaWVzIjpbMSwyXSwidmVuZG9ycyI6W10sInNlcnZpY2VzIjpbXX0=","1year"],["_pandectes_gdpr","eyJzdGF0dXMiOiJjdXN0b20iLCJ0aW1lc3RhbXAiOjE3NDA0MjY0MjksInByZWZlcmVuY2VzIjo2LCJpZCI6IjY3YmNjY2JkYTNmMTg5MGNkMGQ1Mjk2YyJ9","1year"],["Coway_Cookie_Acceptance","Coway_Cookie_Acceptance"],["cookie-consent-settings","{\"tracking\":false,\"marketing\":false}"],["cookieConsent","a100","5256000"],["zdconsent","optin"],["GTM-PRIVACY","{\"adConsentGranted\":false,\"analyticsConsentGranted\":false,\"functionalityConsentGranted\":false,\"personalizationConsentGranted\":false,\"securityConsentGranted\":false,\"mediaConsentGranted\":false,\"expiry\":1859450273208}"],["cookie-agreed","2"],["cookie-agreed-categories","[\"required\",\"video\"]"],["cookies_consent_set","$now$"],["wpconsent_preferences","{\"essential\":true,\"statistics\":false,\"marketing\":false}"],["Tingstad.CookieConsent","StrictlyNecessary","","","reload","1"],["_cc_20240307","{\"functionality_storage\":\"granted\",\"analytics_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_storage\":\"denied\",\"ad_personalization\":\"denied\"}"],["bmf_bstage_accept_cookie","{\"e\":true,\"f\":false,\"a\":false,\"d\":1}"],["crgpd","UUID=1;CUSTOM_USER_ID=;USER_HAS_MADE_CHOICE=true;ANALYTICS_COOKIES=false;FUNCTIONAL_COOKIES=false;ADVERTISING_COOKIES=false"],["cc_pref","{\"approved\":[]}"],["adiva_cookies","{\"success\":true,\"data\":{\"cookies\":{\"neophodniKolacici\":\"true\",\"statistickiKolacici\":\"false\",\"marketinskiKolacici\":\"false\"}}"],["acceptCookiePolicyDate","-1317055050"],["sg_cookie_settings","{\"necessary\":true,\"preferences\":false,\"statistics\":false,\"marketing\":false}"],["_pandectes_gdpr","eyJjb3VudHJ5Ijp7ImNvZGUiOiJGSSIsInN0YXRlIjoiMTUiLCJkZXRlY3RlZCI6MTc0MDkyOTg5M30sInN0YXR1cyI6ImN1c3RvbSIsInRpbWVzdGFtcCI6MTc0MDkyOTkxMiwicHJlZmVyZW5jZXMiOjYsImlkIjoiNjdjNDdiNzgzN2MyYmQzZjQwMTJiMDM4In0=","1year"],["eupubconsent-v2","CQQt_NAQQt_NAAcABBENBnFgAAAAAAAAABpYJQMB3CJ8BSFDYSZ1IIskaIUXwRABxkQhAgLBAwABiBKAOIQAkCAgAABANCACAAIAMHBAAAFADIAAAAAAIAgAIAAMIAAQAABIIABAAAAAAAAQCAgAAABAAQAQgmAEAAcAgAAlgAIoAFAAAABCAACBAAAAEAAFAAEAAAAAAAAAAAAIICwABQAAjAAAAAAAABgQAAAAAAAEAAABoQAIAA4ATgD9wIVhIEQACwAKgAcAA8ACAAGQANAAeABMACkAFUAN4AegA_ACEAEcAJoAYYAygBzgDuAH6AP8AkoBfQEiAJ2AUOAtgBeYDLgGxgQYAheKgBABMAEcMgBABMAEcOADQAOAA8ANAAjgCEAP3AhWOgSAALAAqABwAEAAMgAaAA8ACYAFIAKoAYgA3gB6AD8AJoATgAwwBlADRAHOAO4AfoA_wCLAEdAJKAiwBF4CRAE7AKHAWwAvMBlgDLgIMEAAYADwA0AQrIQDgAFgBMACqAGIAN4AegBHADnAHcAP8AvoCFgEWAQYJAAwB3AMsAfuSgHgALAA4ADwAJgAUgAqgBiAEcAhYBFgCLwEiALYAXmBBgoAJAEcAJwAdwB_wFSAKyAfuBCsCF5SA-AAsACoAHAAQAAyABoADwAJgAUgAqgBiAD8AMoAaIA5wB-gEWAI6ASUAuoB7QELAIvASIAnYBQ4C2AF5gMsAZcBBgsAYAAeABkAFQAQgAkABsAEcAKQAdwBRQCmgF2AMIAZwA04B1QD2gIFAQsAtgBj4DYwICgQwA.YAAAAAAAAAAA","","","domain",".max.com","reload","1"],["cookieSettings","{\"essential\":true,\"functional\":false,\"analytical\":false,\"advertising\":false}","","","reload","1"],["bt_privacy_consent","Agreed_Necessary"],["frank_cookie_settings_v2","{\"consents\":[\"functional\",\"personal\"],\"acceptedAt\":\"1\"}"],["cookie.setting","basic"],["cookieconsent_wig_2","{\"v\":2,\"c\":[],\"lc\":\"1\"}"],["user_cookie_consent","{\"values\":[\"essential\"],\"created_at\":1746326596,\"ttl\":131536000,\"version_name\":\"pl_1.0\",\"consent_id\":\"1\"}"],["cookiebar_accept_functional","$now$"],["cookiebar_accept_version","20231122093052"],["__hs_cookie_cat_pref","1:false,2:false,3:false"],["cookie_preferences","{\"advertising\":false,\"analytics\":false,\"version\":2}"],["onfy_consents","necessary=unbedingt_erforderliche_cookies&targeting=&functional=&performance="],["zoner-cookies-consent-v1","{\"necessary\":true,\"functional\":false,\"analytical\":false,\"marketing\":false}"],["Oru-accept-cookies","oru-denied","","","reload","1"],["_cookienew","acknowledged"],["dsgvo_preferences","{\"cookiebanner\":1,\"dsgvo_statistics\":\"0\",\"dsgvo_marketing\":\"0\",\"dsgvo_external_3rd_party_youtube\":\"1\"}","","reload","1"],["cookieConsentPrefs","{\"cg_essential\":1,\"cg_analytical\":0,\"cg_marketing\":0,\"expire\":182,\"secure\":1}"],["cc_cookie","{\"categories\":[\"necessary\",\"preference\"],\"level\":[\"necessary\",\"preference\"],\"revision\":0,\"data\":null,\"rfc_cookie\":false,\"consent_date\":\"$currentISODate$\",\"consent_uuid\":\"00000000-0000-0000-0000-000000000000\",\"last_consent_update\":\"$currentISODate$\"}","","","domain",".www.drk-hannover.de"],["OptanonConsent","groups=C0001%3A1%2CC0002%3A0%2CC0003%3A1%2CC0004%3A0","","","domain",".wrc.com"],["OptanonAlertBoxClosed","$currentDate$","","","domain",".wrc.com"],["hol_CookiePreferences","{\"necessary\":true,\"analytics\":false,\"advertising\":false}"],["amcookie_allowed","1%2C4","1year"],["entrio_cookie","[{\"title\":\"Necessary\",\"id\":\"REQ\",\"checked\":true},{\"title\":\"Statistics\",\"id\":\"STATS\",\"checked\":false},{\"title\":\"Marketing\",\"id\":\"MARK\",\"checked\":false}]"],["CookieConsent","{necessary:true%2Cpreferences:true%2Cstatistics:false%2Cmarketing:true}","1year"],["CookieConsent","{necessary:true%2Cpreferences:true%2Cstatistics:false%2Cmarketing:false}","1year"],["CookieConsent","{necessary:true%2Cpreferences:true%2Cstatistics:true%2Cmarketing:true}","1year"],["cookie-consent-v2","{\"necessary\":true,\"analytical\":false,\"marketing\":false,\"functional\":true}","1year"],["cookieConsent","[{\"name\":\"essenziell\",\"value\":\"on\"},{\"name\":\"komfort\",\"value\":\"off\"},{\"name\":\"marketing\",\"value\":\"off\"},{\"name\":\"statistik\",\"value\":\"off\"},{\"name\":\"speichern\",\"value\":\"on\"}]"],["sparda.privacySettings","[\"necessary\",\"\",\"\",\"\"]"],["cookie_info","all:0|req:1|track:1|marketing:0|social:0"],["EGDATA_COOKIES_2","eyJnb29nbGVBbmFseXRpY3MiOmZhbHNlLCJzZWxmSG9zdGVkQW5hbHl0aWNzIjp0cnVlLCJnb29nbGVDb25zZW50Ijp7ImFkX3N0b3JhZ2UiOiJkZW5pZWQiLCJhZF91c2VyX2RhdGEiOiJkZW5pZWQiLCJhZF9wZXJzb25hbGl6YXRpb24iOiJkZW5pZWQiLCJhbmFseXRpY3Nfc3RvcmFnZSI6ImRlbmllZCIsImZ1bmN0aW9uYWxpdHlfc3RvcmFnZSI6ImdyYW50ZWQiLCJwZXJzb25hbGl6YXRpb25fc3RvcmFnZSI6ImRlbmllZCIsInNlY3VyaXR5X3N0b3JhZ2UiOiJncmFudGVkIn0sImFocmVmc0FuYWx5dGljcyI6ZmFsc2V9","","","domain",".egdata.app"],["consent_status","analytics:false,personalization:false,marketing:false,necessary:true"],["consent-policy","%7B%22ess%22%3A1%2C%22func%22%3A1%2C%22anl%22%3A0%2C%22adv%22%3A0%2C%22dt3%22%3A1%2C%22ts%22%3A29230412%7D","1year"],["HN-Cookie-Consent","base64:eyJWaWRlb0Nvb2tpZXMiOnRydWUsIkFuYWx5dGljc0Nvb2tpZXMiOmZhbHNlfQ==","","","domain",".helsenorge.no"],["fs-cc","%257B%2522id%2522%253A%2522WsCwWq4mY23eR7nPDUKlP%2522%252C%2522consents%2522%253A%257B%2522analytics%2522%253Afalse%252C%2522essential%2522%253Atrue%252C%2522marketing%2522%253Afalse%252C%2522personalization%2522%253Atrue%252C%2522uncategorized%2522%253Afalse%257D%257D","1year"],["cookieSettings","%7B%22indicatorSeen%22%3Atrue%2C%22analyseCookies%22%3Afalse%2C%22marketingCookies%22%3Afalse%2C%22systemCookies%22%3Afalse%7D"],["DMV_CONSENT","{\"analytics\":false,\"targeting\":false}"],["cookie_consent","{\"necessary\":true,\"targeting\":false,\"performance\":false,\"viewed\":true}"],["_swb","ce58de8a-3d03-4cf7-a8ac-cc8f5573e48f","1year"],["consentMode","{\"ad_storage\":\"denied\",\"analytics_storage\":\"denied\",\"functionality_storage\":\"granted\",\"personalization_storage\":\"denied\",\"ad_user_data\":\"denied\",\"ad_personalization\":\"denied\",\"security_storage\":\"granted\"}"],["didomi_token","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","7889231"],["euconsent-v2","CQaxgwAQaxgwAAHABBENCEFgAAAAAAAAAAZQJpIR_G7dSWljMCJ3YPsEeIwP1dBpQoQgAgCBg2AAjBiU7IQCRWEQAAyKhoAKgQIAoxQBIABFCABEREAAIIAQIAAIAACAABEIICBggFABIWBQAAgOgYohEACAoAJEVBQhkAEAZNI4yViCyAgBihABKWAFAICYAAEEAAI4AAABYMAAImhIlYoAmAJoAEGAABCJAAlBIg4AAAAIAAABBLcAEw0KiCMsgAAIFAwggQACCoIAKBAEAAAQNEAAAYMCnICAC6wiQAgBQACBACAAEGAAIAABIAAIgAgAIBAABAIBAAAAAAABAAQMAAYALAQCAAEB0DFEAAAQJABIyIIFMCAABIICWSoQSAIEFUIAgSwAABATAQAAAAAFAAAALBQCAEgJGJBAExBNAAAQAABRAgAIJCDAABARAAAA.YAAAAAAAAAAA","7889231"],["lvh_cookies_accepted","{%22GB%22:[%22essential%22]}"],["CONSENTMGR","c1:1|c2:0|c3:0|c8:1|ts:$now$|consent:true"]];
const hostnamesMap = new Map([["fandom.com",[0,1,2]],["dailymotion.com",[3,4]],["lefigaro.fr",5],["filmweb.pl",6],["thetrainline.com",[7,46]],["all3dp.com",8],["lidl.*",9],["ellisphere.com",9],["tme.com",9],["threads.com",10],["messenger.com",11],["myscript.com",12],["odido.nl",13],["tele2.nl",13],["bing.com",14],["procvetok.*",15],["stwater.co.uk",16],["x-kom.pl",17],["pizzafan.gr",18],["windhager.eu",19],["banknorwegian.no",20],["banknorwegian.de",20],["plaion.com",21],["apoteket.se",22],["bergbauernmilch.de",22],["berlingske.dk",22],["cineplex.de",22],["danbolig.dk",22],["egmont.com",22],["euroatla.pt",22],["finanzmarktwelt.de",22],["harzwasserwerke.de",22],["hoyavision.com",22],["nos.pt",22],["neue.at",22],["nngroup.com",22],["nordiskfilm.com",22],["toyota-forklifts.se",22],["vn.at",22],["werder.de",22],["werkenbijlidl.nl",22],["wwf.fi",22],["refinery29.com",23],["dasinvestment.com",23],["fof.se",23],["mein-grundeinkommen.de",23],["ekstrabladet.dk",24],["kino.dk",25],["dr.dk",26],["schauspiel.koeln",27],["onlinestempel.ch",28],["chase.co.uk",29],["svt.se",30],["chatreplay.stream",31],["hubblehq.com",32],["hwebber.co.uk",33],["gereedschapcentrum.nl",34],["ontinet.com",35],["fplstatistics.co.uk",36],["softonic.com",37],["softonic.pl",38],["informatech.com",39],["aonsolutions.es",40],["launer.com",41],["bever.nl",42],["oem.no",43],["forums.tomsguide.com",44],["tomsguide.com",45],["cc.com",46],["1001games.com",[46,219,220]],["1001hry.cz",[46,219,220]],["1001jeux.fr",[46,219,220]],["1001jocuri.ro",[46,219,220]],["1001jogos.com.br",[46,219,220]],["1001jogos.pt",[46,219,220]],["1001pelit.com",[46,219,220]],["1001spiele.de",[46,219,220]],["auth.hbomax.com",[46,140]],["elkspel.nl",[46,219,220]],["gamespot.com",[46,161]],["giochixl.it",[46,219,220]],["glamour.com",[46,234]],["grajteraz.pl",[46,219,220]],["isladejuegos.com",[46,219,220]],["jatekokxl.hu",[46,219,220]],["mtvuutiset.fi",[46,155]],["paixnidiaxl.gr",[46,219,220]],["pushsquare.com",[46,185,186]],["spelo.se",[46,219,220]],["spillespill.no",[46,219,220]],["spilxl.dk",[46,219,220]],["thejournal.ie",[46,175]],["vkmag.com",[46,153,154]],["zdnet.com",[46,168]],["zoom.us",[46,241,242]],["mesta.net",[46,245]],["tredz.co.uk",[46,257]],["uphold.com",[46,267,268]],["webuyanycar.com",46],["scarlet.be",[46,666]],["resursbank.fi",47],["ft.com",48],["what3words.com",49],["aufstiegs-bafoeg.de",50],["bafa.de",50],["ble.de",50],["bmbf.de",50],["bne-portal.de",50],["bundespolizei.de",50],["schleswig-holstein.de",50],["verfassungsschutz.de",50],["bfarm.de",50],["ganztagsschulen.org",50],["rki.de",50],["delfi.ee",[51,52,53]],["education.lego.com",54],["lego.com",[55,56]],["planningportal.co.uk",57],["veriff.com",[58,59]],["programme.conventus.de",60],["canva.dev",60],["arena.pl",61],["govirtuo.com",[62,63]],["deal.by",64],["zwenkwielen.net",65],["lastmile.lt",66],["biomedcentral.com",67],["link.springer.com",68],["nature.com",69],["springernature.com",70],["palgrave.com",71],["apress.com",72],["bol.com",73],["abw.by",74],["fotografiska.com",75],["osmanlicakelam.net",76],["open.online",[77,78]],["ilovefreegle.org",[79,80]],["eurostar.com",81],["iskon.hr",[82,83,84]],["virgin.com",85],["tvpworld.com",86],["swiss.com",87],["vm.co.mz",88],["blick.ch",[89,90]],["olympics.com",[90,91]],["ted.com",[90,94]],["bitget.com",[90,100]],["everyeye.it",90],["larousse.fr",[90,287]],["20min.ch",[90,299]],["milwaukeetool.eu",[90,315]],["costcobusinessdelivery.com",[90,319]],["patagonia.com",[90,323]],["max.com",[90,736]],["visme.co",[92,93]],["lebenslauf.com",95],["xing.com",96],["webnode.com",97],["onswingers.com",98],["obi.de",99],["obi-italia.it",99],["obi.ch",99],["obi.at",99],["obi.pl",99],["obi.cz",99],["obi.sk",99],["obi.si",99],["buywholefoodsonline.co.uk",101],["mediapart.fr",102],["zus.pl",[103,104]],["ups.com",[105,106]],["www.saal-digital.de",[107,108]],["gamearena.pl",109],["bambaw.com",[110,111]],["design.com",112],["vous.sncf-connect.com",113],["revolut.com",114],["finnair.com",[115,116]],["cosmotemessageplus.gr",117],["veikkaus.fi",118],["aimigo.coach",119],["gymglish.com",119],["skandia.se",120],["welcomestores.gr",121],["siivous.fi",122],["gallup.com",[123,124]],["github.com",125],["opendoors.org",126],["eduskunta.fi",127],["parliament.fi",127],["riksdagen.fi",127],["hus.fi",128],["tvprofil.com",129],["flibco.com",130],["cencenelec.eu",131],["kick.com",132],["filehippo.com",133],["filehippo.de",134],["filehippo.jp",135],["filehippo.pl",136],["elli.eco",137],["autoliitto.fi",138],["maps-norway.com",139],["trafino.fi",141],["pirha.fi",142],["hvr-amazon.my.site.com",143],["bockscornerbrewery.com",144],["vpd.fi",[145,146]],["alehinta.fi",147],["fyndiq.dk",148],["fyndiq.fi",148],["fyndiq.no",148],["fyndiq.se",148],["16100.fi",149],["mediani.fi",150],["evaair.com",[151,152]],["almamedia.fi",[156,157,158,159,160]],["ampparit.com",[156,157,158]],["arvopaperi.fi",[156,157,158]],["kauppalehti.fi",[156,157,158]],["mikrobitti.fi",[156,157,158]],["talouselama.fi",[156,157,158]],["tekniikkatalous.fi",[156,157,158]],["tivi.fi",[156,157,158]],["uusisuomi.fi",[156,157,158]],["asialadies.de",162],["avladies.de",162],["badeladies.de",162],["behaarteladies.de",162],["bizarrladies.de",162],["busenladies.de",162],["deutscheladies.de",162],["devoteladies.de",162],["dominanteladies.de",162],["erfahreneladies.de",162],["escorts24.de",162],["exklusivladies.de",162],["fkk24.de",162],["grosseladies.de",162],["hobbyladies.de",162],["jungeladies.de",162],["kollegin.de",162],["kussladies.de",162],["ladies.de",162],["latinaladies.de",162],["massierendeladies.de",162],["mollyladies.de",162],["nsladies.de",162],["nymphomaneladies.de",162],["orientladies.de",162],["osteuropaladies.de",162],["piercingladies.de",162],["rasierteladies.de",162],["schokoladies.de",162],["tattooladies.de",162],["tsladies.de",162],["zaertlicheladies.de",162],["zierlicheladies.de",162],["1a-finanzmarkt.de",163],["1a-immobilienmarkt.de",163],["1a-reisemarkt.de",163],["1a-singleboerse.de",163],["1a-stellenmarkt.de",163],["gameinformer.com",164],["christianconcern.com",165],["vogue.co.uk",166],["wired.com",166],["jekabpils.lv",167],["news.sky.com",169],["lippu.fi",[170,171,172]],["starcart.com",173],["sydan.fi",174],["europe1.fr",176],["etsy.com",177],["technopat.net",[178,179]],["just-eat.*",[180,181]],["justeat.it",[180,181]],["lieferando.*",[180,181]],["pyszne.pl",[180,181]],["takeaway.com",[180,181]],["thuisbezorgd.nl",[180,181]],["hemmersbach.com",182],["eurogamer.nl",[183,184]],["eurogamer.es",[183,184]],["eurogamer.net",[183,184]],["eurogamer.pl",[183,184]],["eurogamer.pt",[183,184]],["bt.dk",187],["dlalakierni.pl",[188,189,190]],["officiallondontheatre.com",191],["constantin.film",192],["x.com",193],["mundodeportivo.com",194],["nordax.com",195],["empik.com",[196,197,198,199]],["ubs.com",200],["vicko.gr",201],["call-a-pizza.de",202],["jeuxvideo.com",[203,204]],["player.boom973.com",205],["thegreencity.gr",206],["sandberg.world",207],["breitbart.com",208],["saniweb.nl",209],["bensanitair.nl",209],["sanidirect.nl",209],["uswitch.com",210],["ferienpiraten.ch",211],["holidaypirates.com",211],["piratinviaggio.it",211],["travelpirates.com",211],["urlaubspiraten.at",211],["urlaubspiraten.de",211],["vakantiepiraten.nl",211],["viajerospiratas.es",211],["voyagespirates.fr",211],["wakacyjnipiraci.pl",211],["stadtmobil.de",212],["coinbase.com",[213,214]],["go-e.com",215],["starcar.de",216],["hubsite365.com",217],["notion.so",218],["stergioufamily.gr",221],["goed.at",222],["babolat-tenis.pl",223],["denonmarkabolt.hu",223],["monocerus.pl",223],["plastiflora.pl",223],["s4home.pl",223],["salonydenon.pl",223],["vipera.com.pl",223],["zdrowomania.pl",223],["avstore.pl",224],["bednarek.sklep.pl",224],["byinsomnia.com",224],["foto-tip.pl",224],["handloteka.net",224],["hiprom.pl",224],["meblewypoczynkowe.pl",224],["plantica.pl",224],["produkcjakartonow.pl",224],["supportart.pl",224],["yoclub.pl",224],["audioforum-berlin.de",225],["salonydenon.cz",225],["sj.se",[226,450]],["omgevingsloketinzage.omgeving.vlaanderen.be",227],["snowandrock.com",[228,229]],["zdf.de",230],["microbit.org",231],["coolstuff.*",232],["ab.gr",233],["collectibles.mclaren.com",235],["tobis.de",236],["lekarnaave.cz",237],["lekarnalemon.cz",237],["algarvevacation.net",238],["3sat.de",239],["crossnative.com",240],["trezor.io",243],["rb.cz",244],["mafu.de",246],["paf.com",247],["flip.gr",248],["andersiahotel.pl",249],["tether.to",250],["med.uni-rostock.de",251],["advalvas.vu.nl",252],["chron.com",253],["greenwichtime.com",253],["houstonchronicle.com",253],["mysanantonio.com",253],["seattlepi.com",253],["sfchronicle.com",253],["sfgate.com",253],["thetelegraph.com",253],["timesunion.com",253],["24ur.com",254],["hashop.nl",255],["greencaffenero.pl",256],["getdigitalradio.com",258],["sap.com",[259,260,261]],["oxxio.nl",[262,263,264]],["ouraring.com",265],["seb.lv",266],["seb.lt",266],["seb.ee",266],["as.com",[269,270]],["generali.at",[271,272]],["praxis.nl",273],["brico.be",273],["correos.es",274],["bankinter.com",275],["mishcon.com",[276,548]],["lufthansa.com",277],["anwb.nl",278],["fishingclash.game",279],["bancaditalia.it",280],["kent.gov.uk",281],["wolfswinkel.*",282],["www.aok.de",283],["meine.aok.de",284],["haut-koenigsbourg.fr",285],["copilot.microsoft.com",286],["matkahuolto.fi",288],["wolff-mueller.de",289],["gld.nl",290],["omroepwest.nl",290],["omroepzeeland.nl",290],["omropfryslan.nl",290],["rijnmond.nl",290],["rtvdrenthe.nl",290],["rtvnoord.nl",290],["rtvutrecht.nl",290],["stilord.com",291],["stilord.pl",291],["stilord.de",291],["stilord.fr",291],["esprit.*",292],["ls-tc.de",293],["pieceauto-discount.com",294],["omniaracing.net",294],["onlia.ca",295],["spieljochbahn.at",296],["sogeti.com",297],["transferxl.com",298],["postnl.nl",300],["pharmacieauterive-ropars.mesoigner.fr",301],["seen.es",302],["extrudr.com",303],["vilagitasok.hu",304],["yazio.com",305],["kosik.cz",306],["inps.it",307],["versorgungslandkarte.de",308],["hrlab.de",309],["sazkamobil.cz",310],["svepomoci.cz",311],["leabank.no",312],["tlkhorgaszaruhaz.hu",313],["white.market",314],["mytheresa.com",316],["canatura.com",317],["engaro.sk",317],["engaro.hu",317],["fastnedcharging.com",318],["seznam.cz",320],["issue-council.robertsspaceindustries.com",321],["index.hr",322],["pohjanmaanhyvinvointi.fi",324],["barclays.co.uk",325],["home.barclays",326],["maanmittauslaitos.fi",327],["chip.de",328],["faz.net",329],["welt.de",330],["spiegel.de",331],["t3n.de",332],["t-online.de",333],["autobild.de",334],["bild.de",335],["computerbild.de",336],["pcwelt.de",337],["focus.de",338],["geo.de",339],["familie.de",340],["sport1.de",341],["kino.de",342],["likehifi.de",343],["faszination-fankurve.de",344],["raspberry-pi-geek.de",345],["rtl.de",346],["krzbb.de",347],["n-tv.de",348],["lbc.co.uk",349],["aftonbladet.se",350],["omniekonomi.se",350],["bt.no",350],["vg.no",350],["tedbaker.com",351],["hetzner.com",352],["redbull.com",[353,354]],["vroomly.com",355],["nomonday.pl",356],["foodbooking.com",357],["njspotlightnews.org",358],["wliw.org",358],["thirteen.org",358],["njpbs.org",358],["allarts.org",358],["sportiva.shueisha.co.jp",359],["wpb.shueisha.co.jp",359],["bergans.com",360],["utwente.nl",361],["interactivebrokers.*",362],["shipping-portal.com",363],["caib.es",364],["vidal.fr",365],["1und1.de",366],["tvsyd.dk",367],["suedtirolerjobs.it",368],["kaerntnerjobs.at",368],["tirolerjobs.at",368],["salzburgerjobs.at",368],["wienerjobs.at",368],["posti.fi",[369,370]],["biomarkt.de",371],["doordash.com",372],["gsk.com",373],["ignboards.com",374],["visionapartments.com",375],["lush.com",376],["app.idagio.com",377],["www.idagio.com",378],["pekro.cz",379],["rally-maps.com",380],["skatbank.de",[381,763]],["gardengirls.de",382],["webuildgroup.com",383],["airfrance.*",384],["klm.*",384],["golfdigest.com",385],["lancaster.ac.uk",386],["wavecon.de",387],["glamino.com",[388,389]],["schindelhauerbikes.com",390],["fletcherhotelscheveningen.nl",391],["fletcherlandgoedhotelrenesse.nl",391],["prada.com",392],["falatozz.hu",393],["united-domains.de",394],["prospre.io",395],["joinrs.com",396],["ubi.com",[397,398,399,400,401]],["tk.de",402],["www.tk.de",403],["nizke-napeti.cz.abb.com",404],["lovo.ai",405],["ticketswap.*",406],["automobiles.honda.com",[407,408]],["ivass.it",409],["hammerphones.com",410],["character.ai",411],["figma.com",412],["stripe.com",413],["dave.com",414],["nordnet.fi",415],["sportamore.com",416],["pons.com",417],["ilmatieteenlaitos.fi",418],["krungsriasset.com",419],["redcare.it",420],["lieferando.de",[421,422]],["multitronic.fi",423],["krungsrifinnovate.com",424],["sbk.org",425],["kanga.exchange",426],["pirelli.com",427],["blb-tourismus.de",428],["outdooractive.com",428],["kreuzmayr.at",429],["mikolow.geoportal-krajowy.pl",430],["bencebutor.hu",431],["webtic.it",432],["mywot.com",433],["albert.cz",434],["vierumaki.fi",435],["airam.fi",435],["err.ee",436],["thomasmorris.co.uk",437],["yourvoice.westsussex.gov.uk",438],["fass.se",439],["metzeler.com",440],["outlet-pc.*",441],["miles-and-more.com",442],["postfinance.ch",443],["lcp.fr",444],["sailgp.com",445],["hfm.com",446],["typeform.com",447],["sogeti.*",448],["lisbonsaobentohotel.pt",449],["norwegian.com",451],["klett.de",452],["ringside.ai",453],["muziker.*",454],["ir.ung.edu",455],["hockerty.*",456],["tavid.ee",457],["tavex.*",457],["tavexbullion.co.uk",457],["estadao.com.br",458],["elavon.*",459],["elavon.ie",[459,479]],["alhurra.com",460],["era.be",460],["omnipedia.app",460],["1und1.net",[460,568]],["zdf-studios.com",[460,568]],["anadibank.com",[460,627]],["universalgeneve.com",[460,690]],["ki.se",[460,710]],["caf.fr",460],["imt-mines-ales.fr",[460,723,724]],["grandado.com",461],["rooader.pl",462],["powiat-gniezno.pl",462],["fastweb.it",463],["bandcamp.com",464],["lekarna-bella.cz",[465,579]],["lekarenvkocke.sk",465],["tsb.co.uk",466],["eurofirms.com",467],["clairejoster.com",467],["eurofirmsgroup.com",467],["turvakauppa.com",468],["badgeland.no",468],["stickerland.dk",468],["maandag.com",469],["ing.pl",470],["malighting.com",471],["lightpower.de",471],["tallink.com",472],["toto.nl",473],["jobagent.ch",474],["sumissura.com",475],["analizy.pl",[476,477]],["tylko.com",478],["elavon.co.uk",479],["erfurter-bahn.de",480],["wu.ac.at",480],["horsch.com",480],["dsj.de",480],["warta.pl",480],["terchemicals-cee.com",480],["ter-as.pt",480],["ter-as.com",480],["terfrance.fr",480],["terchemicals.com",480],["teringredients.es",480],["teritalia.com",480],["trexanchemicals.com",480],["ternordic.com",480],["gehring-montgomery.com",480],["teruk.co.uk",480],["terasiapacific.com",480],["podleze-piekielko.pl",480],["leetchi.com",481],["nebius.com",482],["stiwa.com",483],["naturesfinest.pt",484],["tatararazors.com",485],["kubinaut.de",486],["audioboom.com",487],["originalky.cz",488],["otpportalok.hu",489],["abconcerts.be",490],["n-sport.net",491],["komerso.pl",492],["socialpress.pl",492],["allfunds.sg",493],["dolnimorava.cz",494],["tkb.ch",495],["artvertisement.de",496],["womex.com",497],["oem.ee",498],["masmovilpanama.com",499],["top-energie.fr",500],["gov.ie",501],["dogfydiet.com",502],["lta.org.uk",[503,504]],["cremadescalvosotelo.com",505],["landespreis-bw.de",506],["console.anthropic.com",507],["bihealth.org",508],["boerse-stuttgart.de",509],["skagenfunds.is",510],["pemacz.cz",511],["landshut.de",512],["architectes.org",513],["seasungames.com",514],["wwz.ch",515],["vietnamairlines.com",516],["dehn.*",517],["desitek.dk",517],["werkenbijkruidvat.nl",518],["elekta.com",519],["gant.sk",520],["leroymerlin.ro",[521,522]],["pyroweb.de",523],["neplp.lv",524],["otpbank.ro",525],["gesundheit-nordhessen.de",526],["bemmaisseguro.com",[527,528,529]],["thombrowne.com",530],["pitviper.*",530],["sks-germany.com",531],["panzerfux.de",531],["store.godox.pl",532],["elgrafico.com.ar",533],["nvinoticias.com",534],["govern.cat",535],["belabel.cz",536],["titancoop.sm",537],["uebba.com",537],["grabo-balloons.com",537],["epac.it",537],["epatitec.info",537],["malattieautoimmunidelfegato.it",537],["nuovifarmaciepatite.it",537],["trapiantofegato.it",537],["indennizzolegge210.it",537],["cirrosi.com",537],["epatiteb.com",537],["fllifranchini.com",537],["gies.sm",537],["ims.sm",537],["lapitagora.com",537],["verdelab.it",537],["solpelotic.com",537],["agesc.it",537],["amitek.it",537],["omcmachinery.com",537],["maraverbena.com",537],["frigoplanet.it",537],["attiva-mente.info",537],["culturacattolica.it",537],["maestralia.com",538],["rentingcoches.com",538],["lanxshoes.com",539],["swffm.de",540],["eam.de",541],["proteincompany.fi",542],["proteinbolaget.se",542],["proteincompany.no",542],["caiacosmetics.*",542],["reliablecontrols.com",543],["envirodesign.eu",544],["isstbn.cz",544],["pivovarium.cz",544],["iwabijoux.cz",544],["doprava-usti.cz",544],["zshercikova.cz",544],["pilous.hu",544],["aljaoptik.cz",544],["webkamery.online",544],["arco-feed.cz",544],["berghof-systems.cz",544],["dolin.eu",544],["ispop.info",544],["sfe.cz",544],["larok.cz",544],["egrus.cz",544],["omnibooks.cz",544],["kartum.cz",544],["truhlarstvinavratil.cz",544],["sprtec.info",544],["bodysalon.cz",544],["zbrane-slany.cz",544],["thkadan.cz",544],["mastrs.cz",544],["domaciobuv.com",544],["chorvatsko.org",544],["mobilservis.sk",544],["sirenipismasvateho.cz",544],["jirislezak.cz",544],["penzionudvojice.cz",544],["icmjh.cz",544],["zszitomirska.info",544],["ural.cz",544],["whippets-dog.cz",544],["mirdacars.cz",544],["vestax.net",544],["dja.cz",544],["zsamskrovi.cz",544],["futoncompany.co.uk",545],["mbhbank.hu",546],["nebo.app",547],["refunder.*",549],["anywhereesim.com",550],["nadeta.cz",551],["gastronovi.com",552],["gibgas.de",553],["paulaschoice.nl",554],["beobank.be",555],["ether.fi",556],["international.kk.dk",[557,558]],["espares.com",559],["centralthe1card.com",560],["breeam.nl",561],["letec.be",562],["cotswoldoutdoor.com",563],["universcine.com",564],["bbb.org",565],["myqnapcloud.com",566],["dashlane.com",567],["mojomortgages.com",569],["javeahouses.com",570],["apk-vk.at",571],["nmhh.hu",572],["guter-gerlach.de",573],["zsgarwolin.pl",573],["online.munzinger.de",574],["bruns-grosse-groessen.de",575],["pensjonistforbundet.no",576],["kosta.at",577],["karkkainen.com",578],["internetlekarna.cz",579],["lekarnasvjosefa.cz",579],["slimbee.cz",579],["miethke.com",580],["pin-freunde.de",580],["icscards.nl",581],["rodekruis.be",582],["mega.io",583],["chamaeleon.de",584],["bta.ee",585],["akbw.de",586],["parcours-formations.fr",587],["swd-darmstadt.de",588],["cunardcruceros.com",589],["bluelightcard.co.uk",590],["bluelightcard.com.au",590],["atomichub.io",591],["dragnsurvey.com",592],["autonet-group.ro",593],["pillhof.com",593],["suince.es",593],["bitbrain.com",594],["hawle.de",595],["inwerk.de",595],["bioquelle.at",595],["europemetalfil.fr",596],["sufilog.com",596],["debatpublic.fr",597],["institutfrancais.com",598],["blogs.pcsoft.fr",599],["renzgroup.*",600],["casper.network",601],["salusinv.es",602],["moellegaarden.dk",603],["silkeborgcaravancenter.dk",603],["bronderslevcaravan.dk",603],["kais-caravan.dk",603],["le-camping.dk",603],["dancamper.dk",603],["e-kruk.pl",604],["dibaq.com",605],["garante.es",605],["jobsireland.ie",606],["moviepass.com",607],["hitchcoq.de",608],["hitchcoq.order.app.hd.digital",609],["betterhomes.ch",610],["betterhomes.at",610],["betterhomes-international.com",610],["betterhomes.de",610],["musti.no",611],["petenkoiratarvike.com",611],["mustijamirri.fi",611],["arkenzoo.se",611],["vetzoo.*",611],["featherliteladders.com",612],["henleyglobal.com",613],["museum.nl",614],["residentportal.com",615],["glassons.com",616],["fbreader.org",617],["roshtein.com",618],["brightdata.com",619],["birenbach.de",620],["programmes.uliege.be",621],["renault-beier.de",[622,623]],["renault-hoffmann-niedernhausen.de",[622,623]],["ofaj.org",624],["craftserve.com",625],["ticket-onlineshop.com",626],["becker-antriebe.shop",628],["leuchtstark.de",628],["topregal.*",[628,712]],["kunsthauswien.com",629],["bokio.co.uk",630],["bokio.se",630],["outdooradventure.dk",631],["lezhinus.com",632],["rehadat-hilfsmittel.de",633],["rehadat-gkv.de",633],["mercedes-benz.cr",634],["allround-pc.com",635],["toyota.com.ph",636],["sparbankenbergslagen.se",637],["swedbankrobur.se",637],["sparbankenikarlshamn.se",637],["olandsbank.se",637],["atvidabergsspb.se",637],["avsparbank.se",637],["alemssparbank.se",637],["virserumssparbank.se",637],["vimmerbysparbank.se",637],["varbergssparbank.se",637],["wwsparbank.se",637],["vadstenasparbank.se",637],["valdemarsvikssparbank.se",637],["ulricehamnssparbank.se",637],["tjorns-sparbank.se",637],["tidaholms-sparbank.se",637],["tjustbanken.se",637],["sodrahestrasparbank.se",637],["sodradalarnassparbank.se",637],["sormlandssparbank.se",637],["sparbankenvm.se",637],["sparbankensjuharad.se",637],["sparbankengoinge.se",637],["sparbankengotland.se",637],["sparbankeneken.se",637],["sparbankenalingsas.se",637],["sparbankenboken.se",637],["snapphanebygdenssparbank.se",637],["skurupssparbank.se",637],["sidensjosparbank.se",637],["salasparbank.se",637],["roslagenssparbank.se",637],["orustsparbank.se",637],["norrbarke-sparbank.se",637],["mjobackssparbank.se",637],["markarydssparbank.se",637],["ltvsparbank.se",637],["leksandssparbank.se",637],["lekebergssparbank.se",637],["laholmssparbank.se",637],["kindaydresparbank.se",637],["ivetoftasparbank.se",637],["sparbankennord.se",637],["sparbankentanum.se",637],["sparbankentranemo.se",637],["sparbankenskane.se",637],["sparbankenskaraborg.se",637],["sparbankenrekarne.se",637],["sparbankenlidkoping.se",637],["sparbankenenkoping.se",637],["hogsbysparbank.se",637],["sparbanken.se",637],["swedbank.lv",637],["swedbank.cn",637],["swedbank.fi",637],["swedbank.se",637],["swedbank.no",637],["haradssparbanken.se",637],["halsinglandssparbank.se",637],["fryksdalenssparbank.se",637],["ekebysparbank.se",637],["dalsbank.se",637],["bjursassparbank.se",637],["smsparbank.se",637],["falkenbergssparbank.se",637],["awwwards.com",638],["parliament.scot",[639,640]],["theoceancleanup.com",641],["kaasuvalo.fi",642],["kaluste10.fi",642],["keiju.fi",642],["masku.com",642],["nettikaluste.fi",642],["puolenkuunpelit.com",642],["rieker-shop.fi",642],["sydanmerkki.fi",642],["verkkokauppatukku.fi",642],["villihinta.fi",642],["nttgame.com",643],["buyzoxs.de",644],["northernpowergrid.com",645],["censhare.com",646],["africatours.dk",647],["wideroe.no",648],["pre-flight-shopping.com",649],["aghotelspa.com",650],["bookings.rnsportscentre.com",650],["heckenpflanzen-heijnen.de",651],["pqs.app",652],["stadtwerke-bonn.de",653],["inkasso.de",654],["nederlandseloterij.nl",655],["thbaker.co.uk",656],["helloprint.co.uk",657],["tavaratalohurrikaani.fi",[658,659]],["svtplay.se",[660,661]],["ops.alertcalifornia.org",662],["city-kebap-bestellen.de",663],["napakka.fi",664],["thewutangclan.com",665],["vbtverhuurmakelaars.nl",667],["maximehotellisbon.com",668],["realhotelsgroup.com",668],["shopstyle.co.uk",669],["profpneumologia.it",670],["kamensenica.sk",670],["movimentoofficinedelsud.it",670],["diazfloristasestrella.com",670],["cosechavida.com",670],["broncoillustration.com",670],["sumoingenio.com",670],["aligepro.es",670],["muevo.es",670],["tsconline.es",670],["polifast.it",670],["kunterbunte-kinder.de",670],["ledocom.cz",670],["aliciasuarez.net",670],["lamagnalonga.org",670],["micropigmentacioncapilarbcn.com",670],["videogenic.es",670],["grundschule-remagen.de",670],["aceitessatunion.com",670],["textsteine.de",670],["administradorfincasblog.com",670],["grafika-dtp-produkcia.sk",670],["unmardeconstelaciones.com",670],["gymnosport.com",670],["easyhomes.com.es",670],["dosc.es",670],["berevolk.com",670],["calamo.se",670],["elserratplanoles.com",670],["ecoparquets.com",670],["zlotaraczkalublin.pl",670],["congresoscostadelsol.com",670],["virgendefatima.es",670],["patronatpremia.cat",670],["abantos.es",670],["acaluca.org",670],["ad-particles.com",670],["afplr.fr",670],["agiletalon.fr",670],["agiratou.com",670],["aidaromero.com",670],["alligatorbioscience.se",670],["anea.es",670],["asapservicios.net",670],["ayudatranspersonal.com",670],["letrayframe.com",670],["osteofisintegral.es",670],["uco.cat",670],["berliner-haeuser.de",670],["bio-cord.es",670],["biotropica.fr",670],["braeu-stueble.de",670],["brendanoharamp.scot",670],["caleulalia.com",670],["cande-sur-beuvron.com",670],["carlhag.se",670],["casadelaveiga.com",670],["caytas.com.tr",670],["cecjecuador.org.ec",670],["cegef.com",670],["centropolisportivomassari.it",670],["cerai.org",670],["cervosgrup.com",670],["copysud.fr",670],["cortilar.com",670],["desfiladeroediciones.com",670],["dibujos-animados.net",670],["direkt-immobilie.de",670],["drpuigdollers.com",670],["dunamys.inf.br",670],["eb2b.com.pl",670],["eiglaw.com",670],["emalec.com",670],["espaisperconviure.es",670],["euroenvio.com",670],["eurotex.es",670],["farmarsketrhyfuturum.cz",670],["fisiolistic.com",670],["gestenaval.com",670],["glaskogen.se",670],["goia.org.pl",670],["granibier.com",670],["grupoisn.com",670],["gruporhzaragoza.com",670],["handelskammaren.com",670],["helitecnics.com",670],["helux.se",670],["hermanosalcaraz.com",670],["hummer.cz",670],["icld.se",670],["ict-project.it",670],["imprentalaspalmas.com",670],["informamiele.it",670],["amanvida.eu",671],["roblox.com",672],["finya.de",673],["printsimple.eu",674],["hearthis.at",675],["weareethos.com",676],["sex.com",677],["yle.fi",678],["oth-regensburg.de",679],["fh-aachen.de",679],["hfwu.de",679],["hs-coburg.de",679],["hs-esslingen.de",679],["hs-kempten.de",679],["ku.de",679],["puresurfcamps.com",679],["tum.de",679],["uni-tuebingen.de",679],["uni-ulm.de",679],["ryanair.com",680],["kwizard.hr",681],["inyova.ch",681],["inyova.de",681],["syacon.com",681],["robocoast.eu",681],["community.tre.se",682],["captainverify.com",683],["journeesdupatrimoine.culture.gouv.fr",684],["geneanet.org",685],["okosgazdi.hu",686],["ooekultur.at",687],["igmetall.de",[688,689]],["klinik-am-ring.de",691],["onelottery.co.uk",[692,693]],["yourschoollottery.co.uk",[692,693]],["rainbowlottery.co.uk",[692,693]],["zen.com",694],["recettesetcabas.com",695],["ajbell.co.uk",696],["br-dge.to",697],["reifenleader.de",698],["hellowork.com",[699,700]],["cetin.cz",701],["vendeeglobe.org",702],["cocote.com",703],["check24-partnerprogramm.de",704],["tierklinik-elversberg.de",705],["shop.mango.com",706],["partsinmotion.co.uk",707],["vwfs.se",708],["electrope.com",709],["citydack.se",711],["ongeo.pl",713],["honeygain.com",714],["galaxus.*",715],["martelaoutlet.fi",716],["chillamo.com",717],["cowaymega.com",718],["crediteurope.ro",719],["on24.fi",720],["downdetector.*",721],["waven-game.com",722],["mymuesli.com",725],["9to5linux.com",726],["tingstad.com",727],["rajsvitidel.cz",728],["h1key-official.com",729],["artms.mod-haus.shop",729],["ubaldi.com",730],["konstnarsnamnden.se",731],["adiva.hr",732],["argenta.be",733],["archon-studio.com",734],["wilfa.com",735],["apollo.pl",737],["budgetthuis.nl",738],["frankenergy.*",739],["frankenergie.*",739],["inshared.nl",740],["wijzeringeldzaken.nl",741],["youmoney.pl",742],["natgeojunior.nl",[743,744]],["workable.com",745],["amazon.jobs",746],["onfy.de",747],["zonercloud.com",748],["oru.se",749],["sas.fi",750],["sas.se",750],["sas.dk",750],["flysas.com",750],["dr-schweizer-schubert.com",751],["siebenbuergisches-museum.de",751],["eckstein-bandoneon.de",751],["oh-heilbronn.de",751],["boehringer.net",751],["schule-mainhardt.de",751],["museumsfreunde.org",751],["antwerpdiamonds.direct",752],["www.drk-hannover.de",753],["wrc.com",[754,755]],["hukumonline.com",756],["halvathinnat.com",757],["entrio.hr",758],["cosmohairshop.de",759],["storyhouseegmont.com",760],["logistics.vr.fi",761],["pointcool-service.fi",762],["aachener-bank.de",763],["abtsgmuender-bank.de",763],["alxinger-bank.de",763],["bank1saar.de",763],["bankhaus-rsa.de",763],["bensbergerbank.de",763],["bergische-volksbank.de",763],["berliner-volksbank.de",763],["bernhauser-bank.de",763],["bodenseebank.de",763],["bopfinger-bank.de",763],["brandenburgerbank.de",763],["bremischevb.de",763],["bruehlerbank.de",763],["budenheimervb.de",763],["deine-volksbank.de",763],["dervolksbanker.de",763],["dettinger-bank.de",763],["die-raiffeisenbank.de",763],["die-vrbank.de",763],["diebank.de",763],["donau-iller-bank.de",763],["dovoba.de",763],["dvrb.de",763],["echterdinger-bank.de",763],["eckernfoerder-bank.de",763],["elvb.de",763],["ems-vechte.de",763],["eu-banking.de",763],["fehntjer-voba.de",763],["frankfurter-volksbank.de",763],["fs-bank.de",763],["genoba-weil.de",763],["genobamuc.de",763],["genobank-unterallgaeu.de",763],["genobank.de",763],["gestalterbank.de",763],["gladbacher-bank.de",763],["grafschafter-volksbank.de",763],["hagnauer-volksbank.de",763],["hamburger-volksbank.de",763],["hannoversche-volksbank.de",763],["harzer-volksbank.de",763],["hdh-voba.de",763],["heidelberger-volksbank.de",763],["hermsdorfer-kreuz.de",763],["huemmlinger-volksbank.de",763],["huettenberger-bank.de",763],["ihre-vb.de",763],["ihrziel.de",763],["kieler-volksbank.de",763],["landbank-horlofftal.de",763],["leipziger-volksbank.de",763],["levo-bank.de",763],["maerkische-bank.de",763],["meine-rvb.de",763],["meine-volksbank-online.de",763],["meine-vrbank.de",763],["meine-vvb.de",763],["meinebank.de",763],["meinevrbank.de",763],["mendener-bank.de",763],["muenchner-bank.de",763],["myvoba.com",763],["oberlandbank.de",763],["oldenburger-volksbank.de",763],["onstmettinger-bank.de",763],["ostfriesische-volksbank.de",763],["raiba-aindling.de",763],["raiba-aitrang-rud.de",763],["raiba-alsbach.de",763],["raiba-altschweier.de",763],["raiba-bidingen.de",763],["raiba-bobingen.de",763],["raiba-boellingertal.de",763],["raiba-buehlertal.de",763],["raiba-buett-gauk.de",763],["raiba-burgthann.de",763],["raiba-bw.de",763],["raiba-donau-heuberg.de",763],["raiba-elsavatal.de",763],["raiba-flachsmeer.de",763],["raiba-floss.de",763],["raiba-gh.de",763],["raiba-gilching.de",763],["raiba-gr.de",763],["raiba-gretha.de",763],["raiba-grimma.de",763],["raiba-hoechberg.de",763],["raiba-idafehn.de",763],["raiba-kaarst.de",763],["raiba-kms.de",763],["raiba-mittenwald.de",763],["raiba-msp.de",763],["raiba-muc-sued.de",763],["raiba-ndwa.de",763],["raiba-neumarkt-opf.de",763],["raiba-neustadt.de",763],["raiba-oberland.de",763],["raiba-pfaffenhofen.dee",763],["raiba-pfaffenwinkel.de",763],["raiba-rastede.de",763],["raiba-rehling.de",763],["raiba-rhh.de",763],["raiba-rupertiwinkel.de",763],["raiba-seenplatte.de",763],["raiba-smue-stauden.de",763],["raiba-steingaden.de",763],["raiba-suedhardt.de",763],["raiba-tuengental.de",763],["raiba-ueda.de",763],["raiba-wallgau-kruen.de",763],["raiba-welling.de",763],["raiba-westhausen.de",763],["raibacalw.de",763],["raibadirekt.de",763],["raibahallertau.de",763],["raibalauenburg.de",763],["raibamol.de",763],["raibawesermarschsued.de",763],["raibawiesedermeer.de",763],["raiffeisen-volksbank-neustadt.de",763],["raiffeisenbank-bs.de",763],["raiffeisenbank-eifel.de",763],["raiffeisenbank-elztal.de",763],["raiffeisenbank-erlenbach.de",763],["raiffeisenbank-gmund.de",763],["raiffeisenbank-gruibingen.de",763],["raiffeisenbank-hiltenfingen.de",763],["raiffeisenbank-im-breisgau.de",763],["raiffeisenbank-kaiserstuhl.de",763],["raiffeisenbank-kastellaun.de",763],["raiffeisenbank-maitis.de",763],["raiffeisenbank-ottenbach.de",763],["raiffeisenbank-plankstetten.de",763],["raiffeisenbank-regensburg.de",763],["raiffeisenbank-regenstauf.de",763],["raiffeisenbank-rosenstein.de",763],["raiffeisenbank-sinzing.dee",763],["raiffeisenbank-straubing.de",763],["raiffeisenbank-wangen.de",763],["raiffeisenbank-westallgaeu.de",763],["raiffeisenbank-wyhl.de",763],["raiffeisenbankersingen.de",763],["rb-af.de",763],["rb-ah.de",763],["rb-allgaeuerland.de",763],["rb-alw.de",763],["rb-am-kulm.de",763],["rb-anger.de",763],["rb-arge.de",763],["rb-arnstorf.de",763],["rb-as.de",763],["rb-bad-saulgau.de",763],["rb-baunatal.de",763],["rb-bechhofen.de",763],["rb-berghuelen.de",763],["rb-beuerberg.de",763],["rb-biebergrund-petersberg.de",763],["rb-burgstaedt.de",763],["rb-chamer-land.de",763],["rb-chiemgau-nord.de",763],["rb-deggendorf.de",763],["rb-denzlingen-sexau.de",763],["rb-dietersheim.de",763],["rb-dreifranken.de",763],["rb-eching.de",763],["rb-eggenthal.de",763],["rb-ehekirchen.de",763],["rb-elln.de",763],["rb-eschweiler.de",763],["rb-frankenhardt-stimpfach.de",763],["rb-frs.de",763],["rb-fuldaerland.de",763],["rb-geiselhoering.de",763],["rb-grabfeld.de",763],["rb-graevenwiesbach.de",763],["rb-grainet.de",763],["rb-gsd.de",763],["rb-gymnich.de",763],["rb-hardt-bruhrain.de",763],["rb-hessennord.de",763],["rb-hfw.de",763],["rb-hirschau.de",763],["rb-hl.de",763],["rb-holzkirchen-otterfing.de",763],["rb-hs.de",763],["rb-ichenhausen.de",763],["rb-idt.de",763],["rb-kieselbronn.de",763],["rb-kirchweihtal.de",763],["rb-kirtorf.de",763],["rb-knobi.de",763],["rb-koetzting.de",763],["rb-lala.de",763],["rb-lauf.de",763],["rb-lechrain.de",763],["rb-lorup.de",763],["rb-mehr.de",763],["rb-mittelschwaben.de",763],["rb-mn.de",763],["rb-mst.de",763],["rb-muenchen-nord.de",763],["rb-nahe.de",763],["rb-niedere-alb.de",763],["rb-nr.de",763],["rb-oberaudorf.de",763],["rb-om.de",763],["rb-onw.de",763],["rb-opf-jura.de",763],["rb-os.de",763],["rb-parkstetten.de",763],["rb-pfaffenhausen.de",763],["rb-rattiszell-konzell.de",763],["rb-reute-gaisbeuren.de",763],["rb-schaafheim.de",763],["rb-schwaben.de",763],["rb-singoldtal.de",763],["rb-sobland.de",763],["rb-sondelfingen.de",763],["rb-sued-oal.de",763],["rb-thannhausen.de",763],["rb-thurnauerland.de",763],["rb-tofk.de",763],["rb-tuerkheim.de",763],["rb-uz.de",763],["rb-wald.de",763],["rb-wegscheid.de",763],["rb-westeifel.de",763],["rb-wila.de",763],["rb-wl.de",763],["rb-wuestenselbitz.de",763],["rbbai.de",763],["rbbs.de",763],["rbbutjadingenabbehausen.de",763],["rbelbmarsch.de",763],["rbfalkenstein-woerth.de",763],["rbk-haag-gars.de",763],["rbkalbe-bismark.de",763],["rbkk.de",763],["rbleezen.de",763],["rbmehring-leiwen.de",763],["rbnl.de",763],["rbobermain.de",763],["rbok.de",763],["rbopr.de",763],["rbowschlag.de",763],["rbpn.de",763],["rbrodenbach.de",763],["rbscharrel.de",763],["rbseestermuehe.de",763],["rbsum.de",763],["rbtodenbuettel.de",763],["rbui.de",763],["rbuv.de",763],["rheingauer-volksbank.de",763],["rileg.de",763],["rmbeg.de",763],["rv-banken.de",763],["rv-ebe.de",763],["rvb-bad-staffelstein.de",763],["rvb-donauwoerth.de",763],["rvb-hassberge.de",763],["rvbankries.de",763],["rvbfresena.de",763],["rvboderspree.de",763],["rvbvarelnordenham.de",763],["rvbwemding.de",763],["sdk-bockum-hoevel.de",763],["sdkinlamstedt.de",763],["skb-buehlertal.de",763],["skb-gemuenden.de",763],["skb-hammah.de",763],["skb-rheinstetten.de",763],["sob-bank.de",763],["spreewaldbank.de",763],["sylter-bank.de",763],["unsere-raiba.de",763],["unsere-voba.de",763],["v-mn.de",763],["v-vb.de",763],["vb-abi.de",763],["vb-alzey-worms.de",763],["vb-ascheberg-herbern.de",763],["vb-bia.de",763],["vb-bo.de",763],["vb-bocholt.de",763],["vb-bochumwitten.de",763],["vb-brandoberndorf.de",763],["vb-delitzsch.de",763],["vb-eg.de",763],["vb-elsen-wewer-borchen.de",763],["vb-emstal.de",763],["vb-eow.de",763],["vb-flein-talheim.de",763],["vb-gescher.de",763],["vb-grebenhain.de",763],["vb-halle.de",763],["vb-heimbach.de",763],["vb-hm.de",763],["vb-hohenlimburg.de",763],["vb-hohenlohe.de",763],["vb-im-hochsauerland.de",763],["vb-isun.de",763],["vb-iw.de",763],["vb-jerichower-land.de",763],["vb-kempen.de",763],["vb-kirnau-krautheim.de",763],["vb-lauterecken.de",763],["vb-limbach.de",763],["vb-loebau-zittau.de",763],["vb-marl-recklinghausen.de",763],["vb-merz.de",763],["vb-mittelhessen.de",763],["vb-muensterland.de",763],["vb-niers.de",763],["vb-nordhuemmling.de",763],["vb-oberberg.de",763],["vb-rb.de",763],["vb-ruhrmitte.de",763],["vb-sauerland.de",763],["vb-schermbeck.de",763],["vb-selm-bork.de",763],["vb-senden.de",763],["vb-ue-saw.de",763],["vb-ueberherrn.de",763],["vb-vso.de",763],["vb-westenholz.de",763],["vb-westerstede.de",763],["vb3.de",763],["vbank.de",763],["vbbh.de",763],["vbbraunlage.de",763],["vbbremennord.de",763],["vbbs.de",763],["vbdessau.de",763],["vbdonw.de",763],["vbeutin.de",763],["vbfoa.de",763],["vbga.de",763],["vbgebhardshain.de",763],["vbgeest.de",763],["vbghb.de",763],["vbgn.de",763],["vbhalle.de",763],["vbhaseluenne.de",763],["vbheiden.de",763],["vbidr.de",763],["vbimharz.de",763],["vbinswf.de",763],["vbkraichgau.de",763],["vbkrefeld.de",763],["vblastrup.de",763],["vbleos.de",763],["vblh.de",763],["vbloeningen.de",763],["vbnh.de",763],["vbohz.de",763],["vbol.de",763],["vboyten.de",763],["vbplus.de",763],["vbraesfeldunderle.de",763],["vbrb-amg.de",763],["vbrb.de",763],["vbrn.de",763],["vbsdn.de",763],["vbsila.de",763],["vbsolling.de",763],["vbspn.de",763],["vbsprockhoevel.de",763],["vbstendal.de",763],["vbsuedemsland.de",763],["vbu-volksbank.de",763],["vbvechta.de",763],["vbvisbek.de",763],["vbvorpommern.de",763],["vbwinsenermarsch.de",763],["verbundvolksbank-owl.de",763],["vereinigte-vrbk.de",763],["vereinigtevolksbankeg.de",763],["vereinte-volksbank.de",763],["voba-alb.de",763],["voba-aw.de",763],["voba-bl.de",763],["voba-breisgau-nord.de",763],["voba-deisslingen.de",763],["voba-eg.de",763],["voba-ermstal-alb.de",763],["voba-fds.de",763],["voba-glm.de",763],["voba-heuchelheim.de",763],["voba-hoba.de",763],["voba-kw.de",763],["voba-mainspitze.de",763],["voba-mg.de",763],["voba-moeckmuehl.de",763],["voba-msw.de",763],["voba-niedergrafschaft.de",763],["voba-ober-moerlen.de",763],["voba-owd.de",763],["voba-rhein-erft-koeln.de",763],["voba-rheinboellen.de",763],["voba-rll.de",763],["voba-rnh.de",763],["voba-schwanewede.de",763],["voba-seligenstadt.de",763],["voba-sh.de",763],["voba-worpswede.de",763],["voba-zuff.de",763],["vobadhk.de",763],["vobaeg.de",763],["vobakl.de",763],["vobakn.de",763],["vobarl.de",763],["vobawhv.de",763],["vobda.de",763],["volksbank-adelebsen.de",763],["volksbank-aktiv.de",763],["volksbank-albstadt.de",763],["volksbank-allgaeu-oberschwaben.de",763],["volksbank-anroechte.de",763],["volksbank-backnang.de",763],["volksbank-baumberge.de",763],["volksbank-berg.de",763],["volksbank-boenen.de",763],["volksbank-boerde-bernburg.de",763],["volksbank-brawo.de",763],["volksbank-breisgau-markgraeflerland.de",763],["volksbank-brenztal.de",763],["volksbank-buehl.de",763],["volksbank-butzbach.de",763],["volksbank-chemnitz.de",763],["volksbank-daaden.de",763],["volksbank-demmin.de",763],["volksbank-dettenhausen.de",763],["volksbank-dr.de",763],["volksbank-dresden-bautzen.de",763],["volksbank-emstek-essen-cappeln.de",763],["volksbank-erft.de",763],["volksbank-ettlingen.de",763],["volksbank-filder.de",763],["volksbank-franken.de",763],["volksbank-freiburg.de",763],["volksbank-gardelegen.de",763],["volksbank-goeppingen.de",763],["volksbank-hameln-stadthagen.de",763],["volksbank-hamm.de",763],["volksbank-heinsberg.de",763],["volksbank-hellweg.de",763],["volksbank-hochrhein.de",763],["volksbank-immenstadt.de",763],["volksbank-jever.de",763],["volksbank-kassel-goettingen.de",763],["volksbank-kierspe.de",763],["volksbank-kleverland.de",763],["volksbank-koeln-bonn.de",763],["volksbank-kurpfalz.de",763],["volksbank-lahr.de",763],["volksbank-langendernbach.de",763],["volksbank-lindenberg.de",763],["volksbank-luebeck.de",763],["volksbank-magdeburg.de",763],["volksbank-mit-herz.de",763],["volksbank-mittleres-erzgebirge.de",763],["volksbank-mittweida.de",763],["volksbank-muensingen.de",763],["volksbank-neckartal.de",763],["volksbank-niederrhein.de",763],["volksbank-niedersachsen-mitte.de",763],["volksbank-nordkirchen.de",763],["volksbank-nordschwarzwald.de",763],["volksbank-nottuln.de",763],["volksbank-oldenburgland-delmenhorst.de",763],["volksbank-ostlippe.de",763],["volksbank-pfullendorf.de",763],["volksbank-pirna.de",763],["volksbank-plochingen.de",763],["volksbank-pur.de",763],["volksbank-raiffeisenbank-regensburg-schwandorf.de",763],["volksbank-remseck.de",763],["volksbank-rhede.de",763],["volksbank-rhein-ruhr.de",763],["volksbank-riesa.de",763],["volksbank-rietberg.de",763],["volksbank-rot.de",763],["volksbank-rottweil.de",763],["volksbank-sandhofen.de",763],["volksbank-sangerhausen.de",763],["volksbank-schlangen.de",763],["volksbank-staufen.de",763],["volksbank-stuttgart.de",763],["volksbank-sulmtal.de",763],["volksbank-thueringen-mitte.de",763],["volksbank-trier-eifel.de",763],["volksbank-trossingen.de",763],["volksbank-ueberlingen.de",763],["volksbank-ulm-biberach.de",763],["volksbank-ulrichstein.de",763],["volksbank-versmold.de",763],["volksbank-vor-ort.de",763],["volksbank-weschnitztal.de",763],["volksbank-wissmar.de",763],["volksbank-wittenberg.de",763],["volksbank-wittgenstein.de",763],["volksbank-wml.de",763],["volksbank-zwickau.de",763],["volksbankeg.de",763],["volksbankeisenberg.de",763],["volksbanking.de",763],["volksbankinostwestfalen.de",763],["volksbankviersen.de",763],["vr-amberg.de",763],["vr-bank-erding.de",763],["vr-bank-freudenberg-niederfischbach.de",763],["vr-bank-mittelsachsen.de",763],["vr-bank-mr.de",763],["vr-bank-muenchen-land.de",763],["vr-bank-muldental.de",763],["vr-bank-nordrhoen.de",763],["vr-bank-online.de",763],["vr-bank-passau.de",763],["vr-bank-wuerzburg.de",763],["vr-banknordeifel.de",763],["vr-bayernmitte.de",763],["vr-dachau.de",763],["vr-dm.de",763],["vr-ff.de",763],["vr-genobank.de",763],["vr-glauchau.de",763],["vr-heimatbank.de",763],["vr-internet.de",763],["vr-lif-ebn.de",763],["vr-ll.de",763],["vr-mb.de",763],["vr-meissen.de",763],["vr-memmingen.de",763],["vr-mfr.de",763],["vr-miba.de",763],["vr-mio.de",763],["vr-neuburg-rain.de",763],["vr-nopf.cz",763],["vr-nopf.de",763],["vr-obm.de",763],["vr-ofrm.de",763],["vr-partnerbank.de",763],["vr-plus.de",763],["vr-rheinahreifel.de",763],["vr-sf.de",763],["vr-sl-mh.de",763],["vr-teilhaberbank.de",763],["vr-vp.de",763],["vr-werdenfels.de",763],["vr.de",763],["vrb-meinebank.de",763],["vrb-niederschlesien.de",763],["vrb-spangenberg.de",763],["vrb-uckermark-randow.de",763],["vrbank-alsheim.de",763],["vrbank-altenburgerland.de",763],["vrbank-augsburg-ostallgaeu.de",763],["vrbank-bamberg-forchheim.de",763],["vrbank-bayreuth-hof.de",763],["vrbank-brs.de",763],["vrbank-coburg.de",763],["vrbank-eg.de",763],["vrbank-ellwangen.de",763],["vrbank-ffb.de",763],["vrbank-hessenland.de",763],["vrbank-hg.de",763],["vrbank-hsh.de",763],["vrbank-hw.de",763],["vrbank-ihn.de",763],["vrbank-in-holstein.de",763],["vrbank-in-thueringen.de",763],["vrbank-isar-vils.de",763],["vrbank-ke-oa.de",763],["vrbank-lahndill.de",763],["vrbank-lb.de",763],["vrbank-mittelhaardt.de",763],["vrbank-mkb.de",763],["vrbank-mw.de",763],["vrbank-obb-so.de",763],["vrbank-osnordland.de",763],["vrbank-ostalb.de",763],["vrbank-suedoldenburg.de",763],["vrbank-suedpfalz.de",763],["vrbank-suedwestpfalz.de",763],["vrbank-sww.de",763],["vrbank-weimar.de",763],["vrbank-westkueste.de",763],["vrbank-wl.de",763],["vrbank.de",763],["vrbankabd.de",763],["vrbankeg.de",763],["vrbankfulda.de",763],["vrbankgl.de",763],["vrbanking.de",763],["vrbanklm.de",763],["vrbankmecklenburg.de",763],["vrbankmitte.de",763],["vrbanknord.de",763],["vrbanksn.de",763],["vrben.de",763],["vrbfw.de",763],["vrbk.de",763],["vrblausitz.de",763],["vrbprignitz.de",763],["vrbsaale-unstrut.de",763],["vrbsw.de",763],["vrdo.de",763],["vreg.de",763],["vrkt.de",763],["vrla.de",763],["vrnu.de",763],["vrsta.de",763],["vvr-bank.de",763],["vvrb.de",763],["vvrbank-krp.de",763],["waldeck-frankenberger-bank.de",763],["westerwaldbank.de",763],["westkreis.de",763],["winterbacher-bank.de",763],["wvb.de",763],["sparda-a.de",763],["sparda-bank-hamburg.de",763],["sparda-h.de",763],["sparda-sw.de",763],["bbbank.de",763],["bib-fairbanking.de",763],["cvw-privatbank-ag.de",763],["dkm.de",763],["eb.de",763],["edekabank.de",763],["evenordbank.de",763],["hausbank.de",763],["kd-bank.de",763],["ligabank.de",763],["meine-bank-no.de",763],["pax-bkc.de",763],["psd-berlin-brandenburg.de",763],["psd-braunschweig.de",763],["psd-hannover.de",763],["psd-muenchen.de",763],["psd-nuernberg.de",763],["psd-west.de",763],["psdbank-ht.de",763],["skb-badhomburg.de",763],["skbwitten.de",763],["sparda.de",764],["sparda-west.de",764],["sparda-n.de",764],["sparda.at",765],["egdata.app",766],["rikstoto.no",767],["teippimestarit.fi",768],["helsenorge.no",769],["viper.patriotmemory.com",770],["migros.com.tr",771],["destekmerkezi.vestel.com.tr",772],["mrcrayfish.com",773],["southparkstudios.nu",774],["fireflies.ai",775],["kotimikro.fi",[776,777]],["loveholidays.com",778],["firstchoice.co.uk",779]]);
const exceptionsMap = new Map([]);
const hasEntities = true;
const hasAncestors = false;

const collectArgIndices = (hn, map, out) => {
    let argsIndices = map.get(hn);
    if ( argsIndices === undefined ) { return; }
    if ( typeof argsIndices !== 'number' ) {
        for ( const argsIndex of argsIndices ) {
            out.add(argsIndex);
        }
    } else {
        out.add(argsIndices);
    }
};

const indicesFromHostname = (hostname, suffix = '') => {
    const hnParts = hostname.split('.');
    const hnpartslen = hnParts.length;
    if ( hnpartslen === 0 ) { return; }
    for ( let i = 0; i < hnpartslen; i++ ) {
        const hn = `${hnParts.slice(i).join('.')}${suffix}`;
        collectArgIndices(hn, hostnamesMap, todoIndices);
        collectArgIndices(hn, exceptionsMap, tonotdoIndices);
    }
    if ( hasEntities ) {
        const n = hnpartslen - 1;
        for ( let i = 0; i < n; i++ ) {
            for ( let j = n; j > i; j-- ) {
                const en = `${hnParts.slice(i,j).join('.')}.*${suffix}`;
                collectArgIndices(en, hostnamesMap, todoIndices);
                collectArgIndices(en, exceptionsMap, tonotdoIndices);
            }
        }
    }
};

const entries = (( ) => {
    const docloc = document.location;
    const origins = [ docloc.origin ];
    if ( docloc.ancestorOrigins ) {
        origins.push(...docloc.ancestorOrigins);
    }
    return origins.map((origin, i) => {
        const beg = origin.lastIndexOf('://');
        if ( beg === -1 ) { return; }
        const hn = origin.slice(beg+3)
        const end = hn.indexOf(':');
        return { hn: end === -1 ? hn : hn.slice(0, end), i };
    }).filter(a => a !== undefined);
})();
if ( entries.length === 0 ) { return; }

const todoIndices = new Set();
const tonotdoIndices = new Set();

indicesFromHostname(entries[0].hn);
if ( hasAncestors ) {
    for ( const entry of entries ) {
        if ( entry.i === 0 ) { continue; }
        indicesFromHostname(entry.hn, '>>');
    }
}

// Apply scriplets
for ( const i of todoIndices ) {
    if ( tonotdoIndices.has(i) ) { continue; }
    try { trustedSetCookie(...argsList[i]); }
    catch { }
}

/******************************************************************************/

// End of local scope
})();

void 0;
