/*******************************************************************************

    uBlock Origin Lite - a comprehensive, MV3-compliant content blocker
    Copyright (C) 2014-present Raymond Hill

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see {http://www.gnu.org/licenses/}.

    Home: https://github.com/gorhill/uBlock

*/

// ruleset: rus-0

// Important!
// Isolate from global scope

// Start of local scope
(function uBOL_abortCurrentScript() {

/******************************************************************************/

function abortCurrentScript(...args) {
    runAtHtmlElementFn(( ) => {
        abortCurrentScriptFn(...args);
    });
}

function abortCurrentScriptFn(
    target = '',
    needle = '',
    context = ''
) {
    if ( typeof target !== 'string' ) { return; }
    if ( target === '' ) { return; }
    const safe = safeSelf();
    const logPrefix = safe.makeLogPrefix('abort-current-script', target, needle, context);
    const reNeedle = safe.patternToRegex(needle);
    const reContext = safe.patternToRegex(context);
    const extraArgs = safe.getExtraArgs(Array.from(arguments), 3);
    const thisScript = document.currentScript;
    const chain = safe.String_split.call(target, '.');
    let owner = window;
    let prop;
    for (;;) {
        prop = chain.shift();
        if ( chain.length === 0 ) { break; }
        if ( prop in owner === false ) { break; }
        owner = owner[prop];
        if ( owner instanceof Object === false ) { return; }
    }
    let value;
    let desc = Object.getOwnPropertyDescriptor(owner, prop);
    if (
        desc instanceof Object === false ||
        desc.get instanceof Function === false
    ) {
        value = owner[prop];
        desc = undefined;
    }
    const debug = shouldDebug(extraArgs);
    const exceptionToken = getExceptionTokenFn();
    const scriptTexts = new WeakMap();
    const textContentGetter = Object.getOwnPropertyDescriptor(Node.prototype, 'textContent').get;
    const getScriptText = elem => {
        let text = textContentGetter.call(elem);
        if ( text.trim() !== '' ) { return text; }
        if ( scriptTexts.has(elem) ) { return scriptTexts.get(elem); }
        const [ , mime, content ] =
            /^data:([^,]*),(.+)$/.exec(elem.src.trim()) ||
            [ '', '', '' ];
        try {
            switch ( true ) {
            case mime.endsWith(';base64'):
                text = self.atob(content);
                break;
            default:
                text = self.decodeURIComponent(content);
                break;
            }
        } catch {
        }
        scriptTexts.set(elem, text);
        return text;
    };
    const validate = ( ) => {
        const e = document.currentScript;
        if ( e instanceof HTMLScriptElement === false ) { return; }
        if ( e === thisScript ) { return; }
        if ( context !== '' && reContext.test(e.src) === false ) {
            // eslint-disable-next-line no-debugger
            if ( debug === 'nomatch' || debug === 'all' ) { debugger; }
            return;
        }
        if ( safe.logLevel > 1 && context !== '' ) {
            safe.uboLog(logPrefix, `Matched src\n${e.src}`);
        }
        const scriptText = getScriptText(e);
        if ( reNeedle.test(scriptText) === false ) {
            // eslint-disable-next-line no-debugger
            if ( debug === 'nomatch' || debug === 'all' ) { debugger; }
            return;
        }
        if ( safe.logLevel > 1 ) {
            safe.uboLog(logPrefix, `Matched text\n${scriptText}`);
        }
        // eslint-disable-next-line no-debugger
        if ( debug === 'match' || debug === 'all' ) { debugger; }
        safe.uboLog(logPrefix, 'Aborted');
        throw new ReferenceError(exceptionToken);
    };
    // eslint-disable-next-line no-debugger
    if ( debug === 'install' ) { debugger; }
    try {
        Object.defineProperty(owner, prop, {
            get: function() {
                validate();
                return desc instanceof Object
                    ? desc.get.call(owner)
                    : value;
            },
            set: function(a) {
                validate();
                if ( desc instanceof Object ) {
                    desc.set.call(owner, a);
                } else {
                    value = a;
                }
            }
        });
    } catch(ex) {
        safe.uboErr(logPrefix, `Error: ${ex}`);
    }
}

function runAtHtmlElementFn(fn) {
    if ( document.documentElement ) {
        fn();
        return;
    }
    const observer = new MutationObserver(( ) => {
        observer.disconnect();
        fn();
    });
    observer.observe(document, { childList: true });
}

function getExceptionTokenFn() {
    const token = getRandomTokenFn();
    const oe = self.onerror;
    self.onerror = function(msg, ...args) {
        if ( typeof msg === 'string' && msg.includes(token) ) { return true; }
        if ( oe instanceof Function ) {
            return oe.call(this, msg, ...args);
        }
    }.bind();
    return token;
}

function safeSelf() {
    if ( scriptletGlobals.safeSelf ) {
        return scriptletGlobals.safeSelf;
    }
    const self = globalThis;
    const safe = {
        'Array_from': Array.from,
        'Error': self.Error,
        'Function_toStringFn': self.Function.prototype.toString,
        'Function_toString': thisArg => safe.Function_toStringFn.call(thisArg),
        'Math_floor': Math.floor,
        'Math_max': Math.max,
        'Math_min': Math.min,
        'Math_random': Math.random,
        'Object': Object,
        'Object_defineProperty': Object.defineProperty.bind(Object),
        'Object_defineProperties': Object.defineProperties.bind(Object),
        'Object_fromEntries': Object.fromEntries.bind(Object),
        'Object_getOwnPropertyDescriptor': Object.getOwnPropertyDescriptor.bind(Object),
        'Object_hasOwn': Object.hasOwn.bind(Object),
        'RegExp': self.RegExp,
        'RegExp_test': self.RegExp.prototype.test,
        'RegExp_exec': self.RegExp.prototype.exec,
        'Request_clone': self.Request.prototype.clone,
        'String': self.String,
        'String_fromCharCode': String.fromCharCode,
        'String_split': String.prototype.split,
        'XMLHttpRequest': self.XMLHttpRequest,
        'addEventListener': self.EventTarget.prototype.addEventListener,
        'removeEventListener': self.EventTarget.prototype.removeEventListener,
        'fetch': self.fetch,
        'JSON': self.JSON,
        'JSON_parseFn': self.JSON.parse,
        'JSON_stringifyFn': self.JSON.stringify,
        'JSON_parse': (...args) => safe.JSON_parseFn.call(safe.JSON, ...args),
        'JSON_stringify': (...args) => safe.JSON_stringifyFn.call(safe.JSON, ...args),
        'log': console.log.bind(console),
        // Properties
        logLevel: 0,
        // Methods
        makeLogPrefix(...args) {
            return this.sendToLogger && `[${args.join(' \u205D ')}]` || '';
        },
        uboLog(...args) {
            if ( this.sendToLogger === undefined ) { return; }
            if ( args === undefined || args[0] === '' ) { return; }
            return this.sendToLogger('info', ...args);
            
        },
        uboErr(...args) {
            if ( this.sendToLogger === undefined ) { return; }
            if ( args === undefined || args[0] === '' ) { return; }
            return this.sendToLogger('error', ...args);
        },
        escapeRegexChars(s) {
            return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        },
        initPattern(pattern, options = {}) {
            if ( pattern === '' ) {
                return { matchAll: true, expect: true };
            }
            const expect = (options.canNegate !== true || pattern.startsWith('!') === false);
            if ( expect === false ) {
                pattern = pattern.slice(1);
            }
            const match = /^\/(.+)\/([gimsu]*)$/.exec(pattern);
            if ( match !== null ) {
                return {
                    re: new this.RegExp(
                        match[1],
                        match[2] || options.flags
                    ),
                    expect,
                };
            }
            if ( options.flags !== undefined ) {
                return {
                    re: new this.RegExp(this.escapeRegexChars(pattern),
                        options.flags
                    ),
                    expect,
                };
            }
            return { pattern, expect };
        },
        testPattern(details, haystack) {
            if ( details.matchAll ) { return true; }
            if ( details.re ) {
                return this.RegExp_test.call(details.re, haystack) === details.expect;
            }
            return haystack.includes(details.pattern) === details.expect;
        },
        patternToRegex(pattern, flags = undefined, verbatim = false) {
            if ( pattern === '' ) { return /^/; }
            const match = /^\/(.+)\/([gimsu]*)$/.exec(pattern);
            if ( match === null ) {
                const reStr = this.escapeRegexChars(pattern);
                return new RegExp(verbatim ? `^${reStr}$` : reStr, flags);
            }
            try {
                return new RegExp(match[1], match[2] || undefined);
            }
            catch {
            }
            return /^/;
        },
        getExtraArgs(args, offset = 0) {
            const entries = args.slice(offset).reduce((out, v, i, a) => {
                if ( (i & 1) === 0 ) {
                    const rawValue = a[i+1];
                    const value = /^\d+$/.test(rawValue)
                        ? parseInt(rawValue, 10)
                        : rawValue;
                    out.push([ a[i], value ]);
                }
                return out;
            }, []);
            return this.Object_fromEntries(entries);
        },
        onIdle(fn, options) {
            if ( self.requestIdleCallback ) {
                return self.requestIdleCallback(fn, options);
            }
            return self.requestAnimationFrame(fn);
        },
        offIdle(id) {
            if ( self.requestIdleCallback ) {
                return self.cancelIdleCallback(id);
            }
            return self.cancelAnimationFrame(id);
        }
    };
    scriptletGlobals.safeSelf = safe;
    if ( scriptletGlobals.bcSecret === undefined ) { return safe; }
    // This is executed only when the logger is opened
    safe.logLevel = scriptletGlobals.logLevel || 1;
    let lastLogType = '';
    let lastLogText = '';
    let lastLogTime = 0;
    safe.toLogText = (type, ...args) => {
        if ( args.length === 0 ) { return; }
        const text = `[${document.location.hostname || document.location.href}]${args.join(' ')}`;
        if ( text === lastLogText && type === lastLogType ) {
            if ( (Date.now() - lastLogTime) < 5000 ) { return; }
        }
        lastLogType = type;
        lastLogText = text;
        lastLogTime = Date.now();
        return text;
    };
    try {
        const bc = new self.BroadcastChannel(scriptletGlobals.bcSecret);
        let bcBuffer = [];
        safe.sendToLogger = (type, ...args) => {
            const text = safe.toLogText(type, ...args);
            if ( text === undefined ) { return; }
            if ( bcBuffer === undefined ) {
                return bc.postMessage({ what: 'messageToLogger', type, text });
            }
            bcBuffer.push({ type, text });
        };
        bc.onmessage = ev => {
            const msg = ev.data;
            switch ( msg ) {
            case 'iamready!':
                if ( bcBuffer === undefined ) { break; }
                bcBuffer.forEach(({ type, text }) =>
                    bc.postMessage({ what: 'messageToLogger', type, text })
                );
                bcBuffer = undefined;
                break;
            case 'setScriptletLogLevelToOne':
                safe.logLevel = 1;
                break;
            case 'setScriptletLogLevelToTwo':
                safe.logLevel = 2;
                break;
            }
        };
        bc.postMessage('areyouready?');
    } catch {
        safe.sendToLogger = (type, ...args) => {
            const text = safe.toLogText(type, ...args);
            if ( text === undefined ) { return; }
            safe.log(`uBO ${text}`);
        };
    }
    return safe;
}

function shouldDebug(details) {
    if ( details instanceof Object === false ) { return false; }
    return scriptletGlobals.canDebug && details.debug;
}

function getRandomTokenFn() {
    const safe = safeSelf();
    return safe.String_fromCharCode(Date.now() % 26 + 97) +
        safe.Math_floor(safe.Math_random() * 982451653 + 982451653).toString(36);
}

/******************************************************************************/

const scriptletGlobals = {}; // eslint-disable-line
const argsList = [["$","1xbet"],["$","append"],["$","contextmenu"],["$","divWrapper"],["$","get"],["$","init_x_place"],["$","mainContainer"],["$","showPopupextra"],["$","window.location.href"],["JSON.parse"],["JSON.parse","atob"],["Math.floor","AdSense"],["Object.defineProperty","rcBuf"],["XMLHttpRequest","document.querySelectorAll"],["__require","/clickunder/"],["addEventListener","DOMContentLoaded"],["atob","void"],["clickExplorer"],["decodeURIComponent","/63cc63/"],["decodeURIComponent","fromCharCode"],["disableSelection","reEnable"],["document.addEventListener","adsBlocked"],["document.createElement"],["document.createElement","/ru-n4p|ua-n4p|загрузка.../"],["document.createElement","Detected"],["document.createElement","Math.random"],["document.createElement","atob"],["document.createElement","delete window"],["document.createElement","document.head.appendChild"],["document.createElement","redtram"],["document.getElementById","composedPath"],["document.getElementsByTagName","unselectable"],["document.oncontextmenu"],["document.onkeydown"],["document.querySelector","className"],["encodeURIComponent","rcBuf"],["fuckAdBlock","undefined"],["jQuery","backgroundImage"],["redram","/загрузка.../"],["setInterval","reload"],["setTimeout","adblockwarn"],["window.innerWidth","rcBuf"],["window.onload","_uWnd"],["document.getElementsByTagName","AMSP.loadAsset"],["document.querySelectorAll","popMagic"],["$","blockWarnClass"],["$","mimicTopClass"],["dispatchEvent","zoomdecorate"]];
const hostnamesMap = new Map([["shaiba.kz",0],["budport.com.ua",1],["asn.in.ua",2],["penzainform.ru",3],["l2top.ru",4],["tut.by",5],["conversion.im",6],["bez-smenki.ru",7],["positiverecords.ru",[8,34]],["freescreens.ru",9],["imgbase.ru",9],["imgcach.ru",9],["imgclick.ru",9],["payforpic.ru",9],["picclick.ru",9],["picclock.ru",9],["picforall.ru",9],["fenglish.site",10],["mp3spy.cc",10],["draug.ru",11],["modsforwot.ru",11],["agroreview.com",[12,35]],["filmisub.com",[13,19]],["kinofen.net",[13,19]],["gdespaces.com",14],["gdespaces.net",14],["google-cloud.services",14],["spac.me",14],["spac.run",14],["spac.wtf",14],["spac1.com",14],["spac1.net",14],["spac1.org",14],["spaces-blogs.com",14],["spaces.im",14],["spcs-files.xyz",14],["spcs.bio",14],["spcs.me",14],["spcs.network",14],["spcs.pro",14],["spcs.reviews",14],["spcs.social",14],["strip2.club",14],["strip2.in",14],["strip2.xxx",14],["usersporn.com",14],["fastpic.org",[15,25]],["allboxing.ru",16],["pravvest.ru",17],["daz3d.ru",18],["my-expert.ru",[20,31,32]],["mod-wot.ru",21],["krolik.biz",22],["1news.com.ua",23],["365news.biz",23],["4mama.ua",23],["4studio.com.ua",23],["agroter.com.ua",23],["apnews.com.ua",23],["argumentiru.com",23],["asiaplustj.info",23],["autotema.org.ua",23],["autotheme.info",23],["avtodream.org",23],["beauty.ua",23],["chas.cv.ua",23],["cheline.com.ua",23],["cikavosti.com",23],["ck.ua",23],["cn.ua",23],["comments.ua",23],["cvnews.cv.ua",23],["day.kyiv.ua",23],["depo.ua",23],["dv-gazeta.info",23],["dyvys.info",23],["economistua.com",23],["expert.in.ua",23],["fedpress.ru",23],["firtka.if.ua",23],["forpost.media",23],["fraza.com",23],["gazeta1.com",23],["glavnoe24.ru",23],["glavpost.ua",23],["golosinfo.com.ua",23],["gorodkiev.com.ua",23],["grad.ua",23],["greenpost.ua",23],["ifnews.org.ua",23],["inforpost.com",23],["inkorr.com",23],["itechua.com",23],["kh.ua",23],["khersonline.net",23],["kolizhanka.com.ua",23],["kop.net.ua",23],["kr.ua",23],["krymr.com",23],["kurskcity.ru",23],["liga.net",[23,41]],["lvnews.org.ua",23],["mega-music.pro",23],["mi100.info",23],["mind.ua",23],["mistosumy.com",23],["moirebenok.ua",23],["mycompplus.ru",23],["nashbryansk.ru",23],["news24today.info",23],["newsua.one",23],["ngp-ua.info",23],["nnews.com.ua",23],["novavlada.info",23],["novynarnia.com",23],["np.pl.ua",23],["odessa-life.od.ua",23],["ogo.ua",23],["panoptikon.org",23],["pg11.ru",23],["pik.net.ua",23],["pingvin.pro",23],["pl.com.ua",23],["planetanovosti.com",23],["podpricelom.com.ua",23],["politnavigator.net",23],["poltava365.com",23],["portal.lviv.ua",23],["pravdatutnews.com",23],["prm.ua",23],["procherk.info",23],["profootball.ua",23],["promin.cv.ua",23],["radiosvoboda.org",23],["ratel.kz",23],["real-vin.com",23],["reporter.ua",23],["risu.ua",23],["rivne.media",23],["rivnenews.com.ua",23],["rusjev.com",23],["russianshowbiz.info",23],["rv.ua",23],["rvnews.rv.ua",23],["showdream.org",23],["sport-kr.com.ua",23],["strana.news",23],["strana.today",23],["sud.ua",23],["te.ua",23],["telekritika.ua",23],["tenews.org.ua",23],["theageoffootball.com",23],["tverigrad.ru",23],["tverisport.ru",23],["tvoymalysh.com.ua",23],["uainfo.org",23],["uanews.org.ua",23],["uatv.ua",23],["ukranews.com",23],["ukrrain.com",23],["unn.com.ua",23],["vchaspik.ua",23],["viva.ua",23],["vlast.kz",23],["vnn24.ru",23],["volnorez.com.ua",23],["volyninfa.com.ua",23],["volyninfo.com",23],["volynpost.com",23],["volynua.com",23],["vsviti.com.ua",23],["westnews.info",23],["womo.ua",23],["wona.com.ua",23],["wworld.com.ua",23],["zbirna.com",23],["zp.ua",23],["crast.net",24],["kg-portal.ru",25],["nnm-club.me",25],["nnmclub.ro",25],["nnmclub.to",25],["shanson320.ru",26],["lrepacks.net",27],["trainzland.ru",28],["transkarpatia.net",29],["skam.online",30],["brigadtv.ru",32],["castle-serial.ru",32],["ehlita.ru",32],["gameout.ru",32],["itevonklass.ru",32],["izmailovtv.xyz",32],["karateltv.ru",32],["lyucifer.tv",32],["m-z.tv",32],["miyzvuk.net",32],["pokazuha.ru",32],["samomdele.tv",32],["saske.tv",32],["sorvigolovatv.ru",32],["taynyeistiny.ru",32],["transformator220.ru",32],["stalker-mods.clan.su",33],["stalker-mods.su",33],["gwss.ru",36],["hardwareluxx.ru",37],["marieclaire.ua",38],["24boxing.com.ua",39],["bilshe.com",39],["businessua.com",39],["f1analytic.com",39],["f1ua.org.ua",39],["football-ukraine.com",39],["footballgazeta.com",39],["footballtransfer.com.ua",39],["glianec.com",39],["nashamama.com",39],["stravy.net",39],["zdorovia.com.ua",39],["livesport.ws",40],["techmusic.ru",42],["hdrezka.*",43],["rezka.*",43],["pornopuk.com",44],["huyamba.tv",44],["piratam.net",44],["piratca.net",44],["porn720.biz",44],["pornoakt.*",44],["sexitorrent.com",44],["sextor.org",44],["domahatv.com",44],["torrent-pirat.com",44],["xtorrent.net",44],["rapidzona.tv",44],["xxxrip.net",44],["hentai-mood.me",44],["xxxtor.com",44],["hentai-share.one",44],["3dn.ru",44],["minigames.mail.ru",45],["mmminigames.mail.ru",46],["afisha.ru",47],["autorambler.ru",47],["championat.com",47],["eda.ru",47],["gazeta.press",47],["gazeta.ru",47],["lenta.ru",47],["letidor.ru",47],["moslenta.ru",47],["motor.ru",47],["passion.ru",47],["quto.ru",47],["rambler.ru",47],["wmj.ru",47]]);
const exceptionsMap = new Map([["new.fastpic.org",[15,25]]]);
const hasEntities = true;
const hasAncestors = false;

const collectArgIndices = (hn, map, out) => {
    let argsIndices = map.get(hn);
    if ( argsIndices === undefined ) { return; }
    if ( typeof argsIndices !== 'number' ) {
        for ( const argsIndex of argsIndices ) {
            out.add(argsIndex);
        }
    } else {
        out.add(argsIndices);
    }
};

const indicesFromHostname = (hostname, suffix = '') => {
    const hnParts = hostname.split('.');
    const hnpartslen = hnParts.length;
    if ( hnpartslen === 0 ) { return; }
    for ( let i = 0; i < hnpartslen; i++ ) {
        const hn = `${hnParts.slice(i).join('.')}${suffix}`;
        collectArgIndices(hn, hostnamesMap, todoIndices);
        collectArgIndices(hn, exceptionsMap, tonotdoIndices);
    }
    if ( hasEntities ) {
        const n = hnpartslen - 1;
        for ( let i = 0; i < n; i++ ) {
            for ( let j = n; j > i; j-- ) {
                const en = `${hnParts.slice(i,j).join('.')}.*${suffix}`;
                collectArgIndices(en, hostnamesMap, todoIndices);
                collectArgIndices(en, exceptionsMap, tonotdoIndices);
            }
        }
    }
};

const entries = (( ) => {
    const docloc = document.location;
    const origins = [ docloc.origin ];
    if ( docloc.ancestorOrigins ) {
        origins.push(...docloc.ancestorOrigins);
    }
    return origins.map((origin, i) => {
        const beg = origin.lastIndexOf('://');
        if ( beg === -1 ) { return; }
        const hn = origin.slice(beg+3)
        const end = hn.indexOf(':');
        return { hn: end === -1 ? hn : hn.slice(0, end), i };
    }).filter(a => a !== undefined);
})();
if ( entries.length === 0 ) { return; }

const todoIndices = new Set();
const tonotdoIndices = new Set();

indicesFromHostname(entries[0].hn);
if ( hasAncestors ) {
    for ( const entry of entries ) {
        if ( entry.i === 0 ) { continue; }
        indicesFromHostname(entry.hn, '>>');
    }
}

// Apply scriplets
for ( const i of todoIndices ) {
    if ( tonotdoIndices.has(i) ) { continue; }
    try { abortCurrentScript(...argsList[i]); }
    catch { }
}

/******************************************************************************/

// End of local scope
})();

void 0;
